/**
 * Copyright (c) 2024-25 Dieter Steinwedel. All rights reserved.
 *
 * Unauthorized copying and publishing of this file, via any medium is strictly prohibited.
 */

/**
 * Reads the computed style (key) of an element (source). If it does not exists, the fallback is 
 * returned.
 */
function css_var(source, key, fallback) {
	const computed_style = getComputedStyle(source);
	let property = computed_style.getPropertyValue(key);
	if (property && property !== '') return property.trim();
	else return fallback;
}

/**
 * Reads the font size from the given style.
 */
function font_size_in_px(style) {
	return parseFloat(style.fontSize);
}

/**
 * Creates a font string used for canvas elements.
 */
function canvas_font_from(style) {
	const fontSize = style.fontSize;
	const fontFamily = style.fontFamily;
	const fontStyle = style.fontStyle;
	const fontWeight = style.fontWeight;
	
	return fontStyle + ' ' + fontWeight + ' ' + fontSize + ' ' + fontFamily;
}

/**
 * Sets the width and height of an canvas based on its bounding client rectangle.
 * Padding and border are taken into account.
 */
function canvas_set_dimensions(canvas, scale_factor) {
    const rect = canvas.getBoundingClientRect();

    const computedStyle = window.getComputedStyle(canvas);
    const paddingLeft = parseFloat(computedStyle.paddingLeft);
    const paddingRight = parseFloat(computedStyle.paddingRight);
    const paddingTop = parseFloat(computedStyle.paddingTop);
    const paddingBottom = parseFloat(computedStyle.paddingBottom);
    const borderLeft = parseFloat(computedStyle.borderLeftWidth);
    const borderRight = parseFloat(computedStyle.borderRightWidth);
    const borderTop = parseFloat(computedStyle.borderTopWidth);
    const borderBottom = parseFloat(computedStyle.borderBottomWidth);

	const scale = scale_factor || window.devicePixelRatio;
	
    const width = (rect.width - paddingLeft - paddingRight - borderLeft - borderRight) * scale;
    const height = (rect.height - paddingTop - paddingBottom - borderTop - borderBottom) * scale;

	canvas.setAttribute('width', width);
	canvas.setAttribute('height', height);
	
	return scale;
}

/**
 * Searches for the minimal rect with content in the canvas. 
 */
function canvas_content_rect(canvas, threshold = 0, measure_width_and_height = false) {
	let result = { 
		left: 0, 
		top: 0, 
		width: 0, 
		height: 0,
	};
		
	const byte_threshold = Math.max(1, Math.min(threshold * 255, 255));
	
	const width = Math.ceil(parseFloat(canvas.getAttribute('width'))) || 0;
	const height = Math.ceil(parseFloat(canvas.getAttribute('height'))) || 0;
	if (width === 0 || height === 0) return result;

	const byte_array = canvas.getContext("2d").getImageData(0, 0, width, height).data;			
	let found = false;
					
	for (let y = 0; y < height; y++) {
		for (let x = 0; x < width; x++) {		        
			const index = (y * width + x) * 4 + 3;
			if (byte_array[index] >= byte_threshold) {
				found = true;
				if (y === 0) result.top = y;
	            else result.top = y - 1;
				break;
			}
		}
		if (found) break;
	}
		    
	found = false;
	for (let x = 0; x < width; x++) {
		for (let y = 0; y < height; y++) {		
			const index = (y * width + x) * 4 + 3;
			if (byte_array[index] >= byte_threshold) {
			  	found = true;
				result.left = x;
	            break;
			}
		}
	    if (found) break;
	}
	
	if (measure_width_and_height) {
		found = false;
		for (let y = height - 1; y >= 0; y--) {
	    	for (let x = 0; x < width; x++) {		        
				const index = (y * width + x) * 4 + 3;
				if (byte_array[index] >= byte_threshold) {            	
					found = true;
	                result.height = y - result.top + 2;
					break;
				}
			}
			if (found) break;
		}

	    found = false;
	    for (let x = width - 1; x >= 0; x--) {
			for (let y = 0; y < height; y++) {		
				const index = (y * width + x) * 4 + 3;
				if (byte_array[index] >= byte_threshold) {
				  	found = true;
					result.width = x - result.left + 1;
	                break;
				}
			}
	        if (found) break;
		}
	}
	
	return result;
}

var calc_canvas = document.createElement("canvas");
var calc_ctx = calc_canvas.getContext("2d");			 

/**
 * Measures the real size of a single-lined formatted_time.
 */
function canvas_measure_text(text, computedStyle, threshold = 0, measure_width_and_height = false) {
	if (text === undefined || typeof text !== 'string') throw new Error('Parameter "text" not set or its value is not a string.');
	if (!computedStyle === undefined) throw new Error('Parameter "computedStyle" not set or its value is not an instance of a style.');
	if (typeof measure_width_and_height !== 'boolean') throw new Error('Parameter "measure_width_and_height" is not an instance of a boolean.');
				
	const font = canvas_font_from(computedStyle);
	
	calc_ctx.textBaseline = "top";
    calc_ctx.font = font;
	
    const text_metrics = calc_ctx.measureText(text);
	console.log(text_metrics);

	const width = Math.ceil(text_metrics.width);
	const height = Math.ceil(font_size_in_px(computedStyle));			
	
	calc_canvas.width = width;
	calc_canvas.height = height;
	
	calc_ctx.clearRect(0, 0, width, height);
	calc_ctx.textBaseline = "top";
	calc_ctx.font = font;
	
	calc_ctx.fillText(text, 0, 0);
	var result = canvas_content_rect(calc_canvas, threshold, measure_width_and_height);
	result.baseline = -text_metrics.alphabeticBaseline - result.top;
	result.font_size_in_px = font_size_in_px(computedStyle);
	result.font = font;
	return result;
}

// ChannelLayout ===================================================================================

class StreamInfo extends HTMLElement {
	
	#title;
	#channel_bitmap;
	#format;
	#sample_rate;
	#bits;
	
	#dom_initialized;
	#parent_element;
	#resize_observer;
	#last_component_height;
	
	#style_matrix_cell;
	#html_stream_info;
	#html_title;
	#html_matrix;
	#html_signal;
	#html_format;
	#html_sample_rate;
	#html_bits;	
	
	constructor() {
		super();
		this.#channel_bitmap = 0;
		this.#title = "";
		this.#dom_initialized = false;
		this.#last_component_height = -1;
		this.#resize_observer = new ResizeObserver(entries => {
            for (let entry of entries) {
                this.resize();
            }
        });
	}

	get channel_bitmap() {
	    return this.#channel_bitmap;
	}

	set channel_bitmap(new_channel_bitmap) {
	    if (this.#channel_bitmap !== new_channel_bitmap) {
	        this.#channel_bitmap = new_channel_bitmap;
	        this.update_matrix();
	    }
	}
	
	get title() {
	    return this.#title;
	}

	set title(new_title) {
        this.#title = new_title;
        this.update_matrix();
	}
	
	get format() {
	    return this.#format;
	}

	set format(new_format) {
        this.#format = new_format;
        this.update_format();
	}
	
	get sample_rate() {
	    return this.#sample_rate;
	}

	set sample_rate(new_sample_rate) {
        this.#sample_rate = new_sample_rate;
        this.update_sample_rate();
	}
	
	get bits() {
	    return this.#bits;
	}

	set bits(new_bits) {
        this.#bits = new_bits;
        this.update_bits();
	}
	
	is_bit_set(channel_bitmap, position) {
	    let mask = 1 << position;
	    return (channel_bitmap & mask) !== 0;
	}
	
	get_short_channel_name(position) {
		switch (position) {
			case 0: return "FL";
			case 1: return "FR";
			case 2: return "FC";
			case 3: return "LFE";
			case 4: return "BL";
			case 5: return "BR";
			case 9: return "SL";
			case 10: return "SR";
			default: return "";
		}
	}
	
	disconnectedCallback() {
		if (!this.#parent_element) this.#resize_observer.unobserve(this.#parent_element);
	}
	
	connectedCallback() {			
		const shadow = this.attachShadow({ mode: 'open' });
		
		shadow.innerHTML = `
			<style>
				:host {
					--title-font-family: Sans;
					--title-font-weight: normal;
					--title-font-color: white;
					--title-font-size: 3em;
					--title-gap: 1.5em;
					--matrix-font-family: Sans;
					--matrix-font-weight: normal;
					--matrix-off-bg: grey;
					--matrix-off-color: white;
					--matrix-on-bg: limegreen;
					--matrix-on-color: white;
					--signal-font-family: Sans;
					--signal-font-weight: normal;
					--signal-font-color: white;
				}
				
				#component {
					display: flex;
					flex-direction: column;

					justify-content: flex-end;
					align-items: flex-start;
					height: 100%;
				}
				
				#title {
					font-family: var(--title-font-family);
					font-weight: var(--title-font-weight);
					font-size: var(--title-font-size);
					color: var(--title-font-color);
					padding-bottom: var(--title-gap);
				}
				
				#stream_info {
					display: flex;
					flex-direction: rows;

					justify-content: flex-start;
					align-items: flex-end;
				}
								
				#matrix {	
					margin: 0;
					padding: 0;
					border: 0;
					
					display: grid;
					grid-template-columns: 1fr 1fr 1fr;
					grid-template-rows: 1fr 1fr 1fr;
					
					aspect-ratio: 1;					
				}
				
				#matrix div {
					display: flex;
					align-items: center;
					justify-content: center;
				}
				
				.off {
					font-family: var(--matrix-font-family);
					font-weight: var(--matrix-font-weight);
					background: var(--matrix-off-bg);
					color: var(--matrix-off-color);
					transition: background 0.3s ease;
				}
				
				.on {
					font-family: var(--matrix-font-family);
					font-weight: var(--matrix-font-weight);
					background: var(--matrix-on-bg);
					color: var(--matrix-on-color);
					transition: background 0.3s ease;
					/*filter: drop-shadow(0px 0px 3px var(--matrix-on-bg));*/
				}
				
				.unused {
					display: none;	
				}
				
				#signal {
					font-family: var(--signal-font-family);
					font-weight: var(--signal-font-weight);
					color: var(--signal-font-color);
					display: flex;
					flex-direction: column;
					justify-content: center;
				}
			</style>
			
			<div id="component">
				<div id="title"></div>
				<div id="stream_info">
					<div id="matrix">
						<div id="FL">FL</div>	<div id="FC">C</div>		<div id="FR">FR</div>
						<div id="SL">SL</div>	<div id="LFE">LFE</div>		<div id="SR">SR</div>
						<div id="BL">BL</div>	<div class="unused"></div>	<div id="BR">BR</div>
					</div>
					<div id="signal">
						<div id="format"></div>
						<div id="sample_rate"></div>
						<div id="bits"></div>
					</div>
				</div>
			</div>
		`;
		
		this.#html_stream_info = shadow.getElementById('stream_info');
		this.#html_title = shadow.getElementById('title');
		this.#html_matrix = shadow.getElementById('matrix');
		this.#html_signal = shadow.getElementById('signal');
		this.#html_format = shadow.getElementById('format');
		this.#html_sample_rate = shadow.getElementById('sample_rate');
		this.#html_bits = shadow.getElementById('bits');
		
		const styleSheet = shadow.querySelector('style').sheet;
		for (let i = 0; i < styleSheet.cssRules.length; i++) {
			const rule = styleSheet.cssRules[i];
			if (rule.selectorText === '#matrix div') {
				this.#style_matrix_cell = rule.style;
				break;
			}
		}
		
		this.#dom_initialized = true;
		this.resize();
		this.update_title();
		this.update_matrix();
		this.update_format();
		this.update_sample_rate();
		this.update_bits();
		
		this.#parent_element = this.parentElement;
		this.#resize_observer.observe(this.parentElement);
	}
	
	static get observedAttributes() {
		return ['channel_bitmap', 'title', 'format', 'sample_rate', 'bits', 'style', 'class'];
	}

	attributeChangedCallback(name, old_value, new_value) {
		switch (name) {
			case 'channel_bitmap':
				this.#channel_bitmap = parseFloat(new_value);
				this.update_matrix();				
				break;
				
			case 'title':
				this.#title = new_value;
				this.update_title();
				break;
				
			case 'format':
				this.#format = new_value;
				this.update_format();
				break;
				
			case 'sample_rate':
				this.#sample_rate = new_value;
				this.update_sample_rate();
				break;
				
			case 'bits':
				this.#bits = new_value;
				this.update_bits();
				break;
				
			case 'class':
			case 'style':
				this.resize();
				break;
		}
	}
	
	resize(force) {
		if (this.#dom_initialized !== true) return;
		
		this.update_title(false);
		
		const style = getComputedStyle(this);
		const width = parseFloat(style.width);
		const max_height = style.maxHeight === 'none' ? (window.innerHeight / 2) : parseFloat(style.maxHeight);
		const title_height = this.#html_title.clientHeight;
		const matrix_height = Math.min(width * 0.9 / 2, max_height - title_height);		
		const component_height = matrix_height + title_height;
		
		if (force !== true && this.#last_component_height === component_height) return;
		this.#last_component_height = component_height;
		
		const matrix_gap = matrix_height * 0.035;
		const matrix_cell_height = (matrix_height - 2 * matrix_gap) / 3;
		
		this.style.height = component_height + 'px';
		
		this.#html_stream_info.style.gap = width * 0.05 + 'px';
		
		this.#html_signal.style.lineHeight = 1.5;
		this.#html_signal.style.fontSize = (matrix_height / 4.4) + 'px';
		this.#html_signal.style.height = matrix_height + 'px';
		
		this.#html_matrix.style.gap = matrix_gap + 'px';
		this.#html_matrix.style.width = matrix_height + 'px';
		this.#html_matrix.style.height = matrix_height + 'px';
		
		this.#style_matrix_cell.fontSize = (matrix_cell_height * 0.5) + 'px';		
		this.#style_matrix_cell.borderRadius = (matrix_cell_height * 0.1) + 'px';
	}
	
	update_title() {
		if (this.#dom_initialized !== true) return;		
		this.#html_title.textContent = this.#title;
	}
	
	update_format() {
		if (this.#dom_initialized !== true) return;
		const no_signal = typeof this.#format === 'undefined' || this.#format == null || this.#format.toLowerCase() === 'no signal' || this.#format.toLowerCase() === '';
		this.#html_format.innerHTML = !no_signal ? this.#format : '&nbsp;';
	}
	
	update_sample_rate() {
		if (this.#dom_initialized !== true) return;
		const val = parseFloat(this.#sample_rate) / 1000;		
		this.#html_sample_rate.innerHTML = val > 0 ? val + ' kHz' : '&nbsp;';
	}
	
	update_bits() {
		if (this.#dom_initialized !== true) return;
		const bits = parseFloat(this.#bits);		
		this.#html_bits.innerHTML = bits > 0 ? bits + ' Bit' : '&nbsp;';
	}
		
	update_matrix() {	
		if (this.#dom_initialized !== true) return;
		const shadow = this.shadowRoot;
		if (!shadow) return;
		
		this.update_title(true); 
		
		for (let i = 0; i <= 10; i++) {
			let id = this.get_short_channel_name(i);
			if (id === "") continue;
			let channel = shadow.getElementById(id);
			if (!channel) continue;
			if (this.is_bit_set(this.#channel_bitmap, i)) {
				channel.classList.add('on');
				channel.classList.remove('off');
			} else {
				channel.classList.add('off');
				channel.classList.remove('on');				
			}
		}
	}
}

// Slider ==========================================================================================

class Slider extends HTMLElement {
	
	#dom_initialized;
	#touch_area;
	#background;	
	#indicator;
		
	#min;
	#max;
	#value;
	#step;
	
	#orientation;
	
	#user_select;
	
	#is_sliding;
	#ref_pixel;
	#ref_value;
	#pixel_per_step;
	
	#parent_element;
	#resize_observer;
	
		
	constructor() {
		super();
		
		this.#dom_initialized = false;
		this.#min = 0;
		this.#max = 100;
		this.#value = 0;
		this.#step = 1;
		this.#is_sliding = false;
		
		this.#orientation = 0;
		
		this.#parent_element = null;
		this.#resize_observer = new ResizeObserver(entries => {
            for (let entry of entries) {
                this.update_style();
            }
        });
	}
	
	get value() {
        return this.#value;
    }

    set value(new_value) {
        if (this.#value !== new_value) {
            this.#value = new_value;
            this.update_indicator(false);
        }
    }
    
    get min() {
		return this.#min;
	}
	
	set min(new_value) {
        if (this.#min !== new_value) {
            this.#min = new_value;
            this.update_indicator(false);
        }
    }
    
    get max() {
		return this.#max;
	}
	
	set max(new_value) {
        if (this.#max !== new_value) {
            this.#max = new_value;
            this.update_indicator(false);
        }
    }
    
    get step() {
		return this.#step;
	}
	
	set step(new_value) {
        if (this.#step !== new_value) {
            this.#step = new_value;
            this.update_indicator(false);
        }
    }
	
	get is_sliding() {
		return this.#is_sliding;
	}
	
	disconnectedCallback() {
		this.#dom_initialized = false;
		if (!this.#parent_element) this.#resize_observer.unobserve(this.#parent_element);
		this.#parent_element = null;
	}
	
	connectedCallback() {
		const shadow = this.attachShadow({ mode: 'open' });
		
		const style = document.createElement('style');
		style.textContent = `
			.in {
				background: var(--color, lime);
				position: absolute;
				transition: background 0.3s ease;
			}
			.bg {
				background: var(--bg-color, none);
				position: absolute;
				transition: background 0.3s ease;
			}
			.touch-bg {
				background: var(--touch-bg, none);
				position: absolute;
				cursor: pointer;
				transition: background 0.3s ease;
			}			
		`;

		// Füge den Stil-Tag zum Shadow DOM hinzu
		shadow.appendChild(style);
		
		this.#indicator = document.createElement('div');
		this.#indicator.className = "in";
		
		this.#background = document.createElement('div');
		this.#background.className = "bg";
		this.#background.appendChild(this.#indicator);		
		
		this.#touch_area = document.createElement('div');		
		this.#touch_area.className = "touch-bg";		
		this.#touch_area.appendChild(this.#background);
		
		shadow.appendChild(this.#touch_area);
		
		this.#dom_initialized = true;
		
		this.update_style();
		
		this.#is_sliding = false;
		
		const pointer_position = (e) =>  {		
			if (e.type.includes(`touch`)) {
			    const { touches, changedTouches } = e.originalEvent ?? e;
			    const touch = touches[0] ?? changedTouches[0];
				return { x: touch.pageX,  y: touch.pageY };
			} else if (e.type.includes(`mouse`)) {
				return { x: e.clientX, y: e.clientY }
			}
		}
		
		const start_sliding = (e) => {
			if (!e.cancelable) return; // because of scrolling
			
			e.preventDefault();
			this.#is_sliding = true;
			this.#user_select = css_var(document.documentElement, 'user-select', 'auto');
			document.documentElement.style.userSelect = 'none';
			
			if (this.#orientation === 0 || this.#orientation === 180) {
				this.#ref_pixel = pointer_position(e).y;				
			} else {
				this.#ref_pixel = pointer_position(e).x;
			}
			this.#ref_value = this.#value;
			
			this.dispatchEvent(new CustomEvent('slide', {
                detail: { value: this.#value, is_sliding: true },
                bubbles: true,
                composed: true
            }));
        };
		
		const stop_sliding = (e) => {
			if (this.#is_sliding) {
				e.preventDefault();
				document.documentElement.style.userSelect = this.#user_select;
				
				this.dispatchEvent(new CustomEvent('slide', {
	                detail: { value: this.#value, is_sliding: false },
	                bubbles: true,
	                composed: true
	            }));
			}
			this.#is_sliding = false;
        };
		
		const sliding = (e) => {
	        if (this.#is_sliding) {			
				let current_pixel;
				let delta_pixel;				
				let indicator_size;
				let indicator_offset;
				
				switch(this.#orientation) {
					case 0:
						current_pixel = pointer_position(e).y;
						delta_pixel = this.#ref_pixel - current_pixel;
						break;
						
					case 90:
						current_pixel = pointer_position(e).x;
						delta_pixel = current_pixel - this.#ref_pixel;
						break;
						
					case 180:
						current_pixel = pointer_position(e).y;
						delta_pixel = current_pixel - this.#ref_pixel;
						break;
						
					case 270:
						current_pixel = pointer_position(e).x;
						delta_pixel = this.#ref_pixel - current_pixel;
						break;
				}
					
				let delta_steps = delta_pixel / this.#pixel_per_step;
				if (delta_steps > 0) delta_steps = Math.floor(delta_steps);
				else delta_steps = Math.ceil(delta_steps);
				
				let new_value = this.#ref_value + delta_steps * this.#step;
				
				if (new_value > this.#max) {
					this.#ref_pixel =  current_pixel;
					this.#ref_value = this.#max;
					new_value = this.#max;
				} else if (new_value < this.#min) {
					this.#ref_pixel =  current_pixel;
					this.#ref_value = this.#min;
					new_value = this.#min;					
				}
				
				if (new_value !== this.#value) {
					this.#value = new_value;
					this.update_indicator(true);
				}
			}
		};
		
		this.#touch_area.addEventListener('mousedown', start_sliding);
		this.#touch_area.addEventListener('touchstart', start_sliding);
        document.addEventListener('mouseup', stop_sliding);
		document.addEventListener('touchend', stop_sliding);
        document.addEventListener('mousemove', sliding);
		document.addEventListener('touchmove', sliding);
		
		this.#parent_element = this.parentElement;
		this.#resize_observer.observe(this.#parent_element);
	}
	
	static get observedAttributes() {
		return ['min', 'max', 'value', 'step', 'style', 'class'];
	}

	attributeChangedCallback(name, old_value, new_value) {
		switch (name) {
			case 'min':
				this.#min = parseFloat(new_value);
				this.update_indicator();
				break;
			
			case 'max':
				this.#max = parseFloat(new_value);
				this.update_indicator();
				break;
				
			case 'step':				
				this.#step = parseFloat(new_value);
				this.update_indicator();
				break;
				
			case 'value':
				this.#value = parseFloat(new_value);
				this.update_indicator();
				break;
			
			case 'class':
			case 'style':
				this.update_style();
				break;
		}	
	}
	
	update_indicator(trigger_event) {		
		if (!this.#dom_initialized) return;
		
		let step = Math.max(this.#step, 0);
		let min = this.#min;
		let max = Math.max(this.#min + step, this.#max);
		let value = this.#value;
		if (step > 0) {
			max = Math.floor((max - min) / step) * step + min;
			let steps = Math.round((value - min) / step);
			value = steps * step + min;
		}
		value = Math.min(max, Math.max(min, value));
		let coefficient = (value - min) / (max - min);
		
		switch (this.#orientation) {
			case 0:
				// top, width, height, left
				max = this.#background.getBoundingClientRect().height;
				let top =  max - max * coefficient;
				this.#indicator.style.clipPath = 'rect(' + top + 'px 100% 100% 0px)';
				break;
			
			case 90:
				// top, width, height, left
				let width =  100 * coefficient;
				this.#indicator.style.clipPath = 'rect(0px ' + width + '% 100% 0px)';
				break;
						
			case 180:
				// top, width, height, left
				let height =  100 * coefficient;
				this.#indicator.style.clipPath = 'rect(0px 100% ' + height + '% 0px)';
				break;
				
			case 270:
				// top, width, height, left
				max = this.#background.getBoundingClientRect().width;
				let left =  max - max * coefficient;
				this.#indicator.style.clipPath = 'rect(0px 100% 100% ' + left + 'px)';
				break;
		}
				
		if (trigger_event) {
			this.dispatchEvent(new CustomEvent('input', {
                detail: { value: this.#value, is_sliding: this.#is_sliding },
                bubbles: true,
                composed: true
            }));
		}
	}
	
	update_style() {
		if (!this.#dom_initialized) return;
		
		const ref_width = css_var(this, '--width', '100%');
		const ref_height = css_var(this, '--height', '100%');
		let bg_style = this.#background.style;
		let touch_style = this.#touch_area.style;
		let in_style = this.#indicator.style; 
				
		let touch_x = css_var(this, '--touch-x', '0px');
		let touch_y = css_var(this, '--touch-y', '0px');
		
		touch_style.left = 'calc(-1 * ' + touch_x + ')';
		touch_style.top = 'calc(-1 * ' + touch_y + ')';			
		touch_style.width = 'calc(' + ref_width + ' + 2 * ' + touch_x + ')';
		touch_style.height = 'calc(' + ref_height + ' + 2 * ' + touch_y + ')';
		
		bg_style.left = 'calc(' + touch_x + ')';
		bg_style.top = 'calc(' + touch_y + ')';
		bg_style.width = 'calc(' + ref_width + ' - 2 * ' + touch_x + ')';
		bg_style.height = 'calc(' + ref_height + ' - 2 * ' + touch_y + ')';
		
		in_style.width = ref_width;
		in_style.height = ref_height;
		
		switch (css_var(this, '--orientation', '0')) {
			case '0':
			case '0deg':
			case 'bt':
			case 'to top':
				this.#orientation = 0;
				this.#pixel_per_step = this.#background.getBoundingClientRect().height / (this.#max - this.#min) * this.#step;
				bg_style.borderRadius = 'calc(' + this.#background.getBoundingClientRect().width + 'px / 2)';
				break;
				
			case '90':
			case '90deg':
			case 'lr':
			case 'to right': 
				this.#orientation = 90;
				this.#pixel_per_step = this.#background.getBoundingClientRect().width / (this.#max - this.#min) * this.#step;
				bg_style.borderRadius = 'calc(' + this.#background.getBoundingClientRect().height + 'px / 2)';
				break;
			
			case '180':
			case '180deg':
			case 'tb':
			case 'to bottom': 
				this.#orientation = 180;
				this.#pixel_per_step = this.#background.getBoundingClientRect().height / (this.#max - this.#min) * this.#step;
				bg_style.borderRadius = 'calc(' + this.#background.getBoundingClientRect().width + 'px / 2)';		
				break;
				
			case '270':
			case '270deg':
			case 'rl':
			case 'to left': 
				this.#orientation = 270;
				this.#pixel_per_step = this.#background.getBoundingClientRect().width / (this.#max - this.#min) * this.#step;
				bg_style.borderRadius = 'calc(' + this.#background.getBoundingClientRect().height + 'px / 2)';
				break;				
		}
		
		in_style.borderRadius = bg_style.borderRadius;
		
		this.update_indicator(false);
	}
}

// CanvasText ======================================================================================

class CanvasText extends HTMLElement {
    
	#canvas;
	#ctx;
	
	#observer;
	
	#trim_start;
	#trim_end;

	#dom_initialized;
	
	constructor() {
		super();
		const shadow = this.attachShadow({ mode: 'open' });
		this.#canvas = document.createElement('canvas');
		this.#ctx = this.#canvas.getContext('2d');
		this.#trim_start = true;
		this.#trim_end = true;
		this.#observer = null;
		this.#dom_initialized = false;
		shadow.appendChild(this.#canvas);
    }
	
	get trimStart() {
	    return this.#trim_start;
	}

	set trimStart(new_value) {
	    if (this.#trim_start !== new_value) {
	        this.#trim_start = new_value;
	        this.update_info();
	    }
	}
	
	get trimEnd() {
	    return this.#trim_end;
	}

	set trimEnd(new_value) {
	    if (this.#trim_end !== new_value) {
	        this.#trim_end = new_value;
	        this.update_info();
	    }
	}
	
	static get observedAttributes() {
		return ['trimStart', 'trimEnd', 'style', 'class'];
	}

	attributeChangedCallback(name, old_value, new_value) {
		switch(name) {
			case 'trimStart':
				this.#trim_start = new_value;
				break;
				
			case 'trimEnd':
				this.#trim_end = new_value;
				break;	
		}
		console.log(">>>> attributeChangedCallback: " + name);
		this.update_canvas();
	}

	connectedCallback() {		
		/*this.#observer = new ResizeObserver(entries => {
			for (let entry of entries) {
				console.log(">>>> Observer");
				this.update_canvas();
			}
		});*/
		
		console.log(">>>>  connected");
		document.fonts.ready.then(() => {
			this.#dom_initialized = true;
			//this.#observer.observe(this.parentElement);
			this.update_canvas();	
		});
    }
	
	disconnectedCallback() {
		this.#dom_initialized = false;
		this.#observer.disconnect();
    }
	
	update_canvas() {
		if (this.#dom_initialized === false) return;
		console.log(">>>> update_canvas");
		let style = window.getComputedStyle(this);
		if (style === undefined) return;
		
		let text = this.textContent;
				
		if (this.#trim_start) text = text.trimStart();
		if (this.#trim_end) text = text.trimEnd();

		const rect = canvas_measure_text(text, style, 0.25, true);		
		
		this.#canvas.style.width = rect.width + 'px';
		this.#canvas.style.height = rect.height + 'px';
		console.log("Text dims:");
		console.log(rect);
		
		const scale_factor = canvas_set_dimensions(this.#canvas);
		this.#ctx.scale(scale_factor, scale_factor);
		
		//TODO copy border, margin, padding, background?
		this.#canvas.style.position = style.position;
		this.#canvas.style.top = style.top;
		this.#canvas.style.left = style.left;
		
		this.#canvas.style.border = "1px solid green";
		this.#ctx.textBaseline = "top";
		this.#ctx.font = canvas_font_from(style);
		this.#ctx.fillStyle = style.color;
		this.#ctx.fillText(text, -rect.left, -rect.top);
	}
}


// Digital Clock ===================================================================================

class DigitalClock extends HTMLElement {

	#show_seconds;
	#show_date;
	#show_24h;
	#replace_zeros;
	
	#started;
	#timer_id;
	
	#dom_initialized;
	#dom_time;
	#dom_date;
	
	constructor() {
		super();		
		this.#show_seconds = false;
		this.#show_date = true;
		this.#show_24h = true;
		this.#dom_initialized = false;
		this.#replace_zeros = false;
		
		const shadow = this.attachShadow({ mode: 'open' });
				
				shadow.innerHTML = `
					<style>
						:host {
							--time-color: white;
							--time-font-family: Sans;
							--time-font-weight: normal;
							--time-font-size: 3em;
							
							--date-color: grey;
							--date-font-family: Sans;
							--date-font-weight: normal;
							--date-font-size: 2em;
						}
						
						#clock {
							width: 100%;
							height: 100%;
							
							display: flex;
							flex-direction: column;
							justify-content: center;
							align-items: center;
						}
						
						#time {
							color: var(--time-color);
							font-family: var(--time-font-family);
							font-weight: var(--time-font-weight);
							font-size: var(--time-font-size);
							white-space: nowrap;
							overflow: hidden;
							text-overflow: ellipsis;
						}
						
						#date {
							color: var(--date-color);
							font-family: var(--date-font-family);
							font-weight: var(--date-font-weight);
							font-size: var(--date-font-size);
							white-space: nowrap;
							overflow: hidden;
							text-overflow: ellipsis;
						}
						
						.d-none {
							display: none;
						}
					</style>
					
					<div id="clock" role="timer" aria-live="polite">
						<div id="time"></div>
						<div id="date"></div>
					</div>
				`;

				this.#dom_time = shadow.getElementById('time');
				this.#dom_date = shadow.getElementById('date');
    }
	
	get show_seconds() {
	    return this.#show_seconds;
	}

	set show_seconds(new_value) {
	    if (this.#show_seconds !== new_value) {
	        this.#show_seconds = new_value;
			this.#update_classes();
	        this.#update_clock();
	    }
	}
	
	get show_date() {
	    return this.#show_date;
	}

	set show_date(new_value) {
	    if (this.#show_date !== new_value) {
	        this.#show_date = new_value;
	        this.#update_clock();			
	    }
	}
	
	get show_24h() {
	    return this.#show_24h;
	}

	set show_24h(new_value) {
	    if (this.#show_24h !== new_value) {
	        this.#show_24h = new_value;
			this.#update_classes();
	        this.#update_clock();			
	    }
	}
	
	get replace_zeros() {
	    return this.#replace_zeros;
	}

	set replace_zeros(new_value) {
	    if (this.#replace_zeros !== new_value) {
	        this.#replace_zeros = new_value;
	        this.#update_clock();
	    }
	}
		
	static get observedAttributes() {
		return ['show_seconds', 'show_date', 'show_24h', 'replace_zeros'];
	}

	attributeChangedCallback(name, old_value, new_value) {
		switch(name) {
			case 'show_seconds':
				this.#show_seconds = new_value.toLowerCase() == 'true' || new_value.toLowerCase() == 'yes';
				break;
				
			case 'show_date':
				this.#show_date = new_value.toLowerCase() == 'true' || new_value.toLowerCase() == 'yes';
				if (this.#show_date) this.#dom_date.classList.remove('d-none');
				else this.#dom_date.classList.add('d-none');
				break;
				
			case 'show_24h':
				this.#show_24h = new_value.toLowerCase() == 'true' || new_value.toLowerCase() == 'yes';
				break;
				
			case 'replace_zeros':
				this.#replace_zeros = new_value.toLowerCase() == 'true' || new_value.toLowerCase() == 'yes';
				break;
		}
		
		this.#update_classes();
		this.#update_clock();
	}

	disconnectedCallback() {
		this.#dom_initialized = false;
	}
		
	connectedCallback() {
		this.#dom_initialized = true;
    }
	
	disconnectedCallback() {
		this.#dom_initialized = false;
    }
	
	start() {
		this.#started = true;
		this.#update_clock();
		
		this.#update_classes();
		
		this.#dom_time.classList.remove('d-none');
		
		if (this.#show_date) this.#dom_date.classList.remove('d-none');
		else this.#dom_date.classList.add('d-none');
	}
	
	stop() {
		if (this.#timer_id) {
			clearTimeout(this.#timer_id);
			this.#timer_id = null;
			this.#started = false;
			
			this.#dom_time.classList.add('d-none');
			this.#dom_date.classList.add('d-none');
	    }
	}
	
	#update_classes() {
		if (this.#show_seconds) this.classList.add('seconds');
		else this.classList.remove('seconds');
		
		if (this.#show_24h) this.classList.remove('period');
		else this.classList.add('period');
		
		if (this.#show_date) this.classList.remove('date');
		else this.classList.add('date');
	}
	
	#to_2_digits(n) {
	    return n < 10 ? '0' + n : '' + n;
	}
	
	#update_clock() {
		if (this.#dom_initialized === false || this.#started === false) return;
		
		const now = new Date();
	    
		let hour = now.getHours();
	    const min = this.#to_2_digits(now.getMinutes());
	    const sec = this.#to_2_digits(now.getSeconds());
		let formatted_time;
		
		if (this.#show_24h) {
			formatted_time = this.#show_seconds ? `${hour}:${min}:${sec}` : `${hour}:${min}`;
		} else {
			const period = hour >= 12 ? 'PM' : 'AM';
			hour = hour % 12 || 12;
			formatted_time = this.#show_seconds ? `${hour}:${min}:${sec} ${period}` : `${hour}:${min} ${period}`;
		}

		this.#dom_time.textContent = this.#replace_zeros ? formatted_time.replace(/0/g, 'O') : formatted_time;
		
		if (this.#show_date) {
			const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
			const formatted_date = new Intl.DateTimeFormat(navigator.language, options).format(now);
			this.#dom_date.textContent = this.#replace_zeros ? formatted_date.replace(/0/g, 'O') : formatted_date;
			this.#dom_date.classList.remove('d-none');
		} else {
			this.#dom_date.classList.add('d-none');
		}
	    
	    const delay = this.#show_seconds ? 1000 - now.getMilliseconds() : ((60 - now.getSeconds()) * 1000) - now.getMilliseconds();
	    this.#timer_id = setTimeout(() => this.#update_clock(), delay);
	}
}

// Carousel ========================================================================================

class Carousel extends HTMLElement {
    
    #blocked;
    
    #html_initialized;
    #html_carousel;
    
    constructor() {
		super();
		this.#blocked = false;
		this.#html_initialized = false;
	}
	
	disconnectedCallback() {
		this.#html_initialized = false;
	}
    
    connectedCallback() {
		if (this.#html_initialized) return;
		
		this.#html_carousel = document.createElement('div');		
		this.#html_carousel.style.overflow = 'auto';
		this.#html_carousel.style.width = '100%';

		this.appendChild(this.#html_carousel);

		this.#html_initialized = true;
	}
    
    #add_slide(content, left) {
		const new_slide = document.createElement('div');
		
		if (typeof content === 'string') {
	        new_slide.innerHTML = content;
	    } else if (content instanceof Node) {
	        new_slide.appendChild(content);
	    } else return null;
		
        new_slide.style.left = left;
		new_slide.className = 'hifidom-slide';
		new_slide.style.position = 'absolute';
		new_slide.style.width = '100%';
		new_slide.style.height = '100%';
		new_slide.style.transition = 'left var(--transition-duration) ease';
        this.#html_carousel.appendChild(new_slide);
        
        return new_slide;
	}
    
    set_slide(content) {
		if (this.#blocked || !this.#html_initialized) return null;
		this.#blocked = true;
		
		let result = this.#add_slide(content, '0%');
        
        this.#blocked = false;
        return result;
	}
	
	remove_slides() {
		this.#html_carousel.innerHTML = "";
	}
	
	slide_to(content, forward) {
		if (!this.#html_initialized) return false;
		const old_slide = this.querySelector('.hifidom-slide');
				
		if (!old_slide) return this.set_slide(content);
		
		if (this.#blocked) return false;
		this.#blocked = true;
		
		const new_slide = this.#add_slide(content, (forward ? '100%' : '-100%'));
		        
        setTimeout(() => {
			new_slide.style.left = '0%';
			old_slide.style.left = (forward ? '-100%' : '100%');
						
			old_slide.addEventListener('transitionend', () => {
				this.#html_carousel.removeChild(old_slide);
				this.#blocked = false;
			});
		}, 0);
        
        return new_slide;
	}
}

// VU Meter ========================================================================================

class VUMeter extends HTMLElement {

	#MIN_LEVEL = 0;
	
	#dom_initialized;
	#parent_element;
	#resize_observer;	
	#html_measure;
	#style_observer;
	
	#app;
	#background;
	#bars;
	#scale;
	#fps;
	
	#width;
	#height;
		
	#levels;
	#levels_ts;
	#level_freeze;
	#peak_levels;
	#peak_ts;
	#peak_freeze;
	#channels;
	#sections;
	#bar_thickness;
	#meter_height;
	#scale_lines;
	
	#style_normal_on;
	#style_critical_on;		
	
	#frames_cnt;
	#frames_ts;
	#frames_fps;
		
	constructor() {
		super();
		this.#dom_initialized = false;
		this.#resize_observer = new ResizeObserver(entries => {
            for (let entry of entries) {				
				const rect = entry.target.getBoundingClientRect();
				
				const style = window.getComputedStyle(this.#parent_element);
				const padding_top = parseFloat(style.paddingTop) * 1;
				const padding_bottom = parseFloat(style.paddingBottom) * 1;
				const padding_left = parseFloat(style.paddingLeft);
				const padding_right = parseFloat(style.paddingRight);
						
				const width = Math.max(1, rect.width - padding_left - padding_right);
				const height = Math.max(1, rect.height - padding_top - padding_bottom);	
				
				this.style.width = width + "px";
				this.style.height = height + "px";				
				this.#app.renderer.resize(width, height);
                this.#resize();				
            }
        });
		this.#style_observer = new MutationObserver(() => this.#resize());
				
		this.#frames_cnt = 0;
		this.#frames_ts = 0;
		this.#frames_fps = 0;
		this.#scale_lines = [];
		
		this.#channels = [ 
			{ name: "FL" , start_px: 0 },
			{ name: "FR" , start_px: 0 },
			{ name: "C"  , start_px: 0 },
			{ name: "LFE", start_px: 0 }, 
			{ name: "SL" , start_px: 0 },
			{ name: "SR" , start_px: 0 },
			{ name: "BL" , start_px: 0 },
			{ name: "BR" , start_px: 0 }
		];
		this.#sections = [
			{ lower_bound:  -6, upper_bound:   0, segments: 2, critical:  true, start_px: 0, end_px: 0, decay_per_ms: 0, total_db: 0, total_px: 0 },
			{ lower_bound: -24, upper_bound:  -6, segments: 3, critical: false, start_px: 0, end_px: 0, decay_per_ms: 0, total_db: 0, total_px: 0 },
			{ lower_bound: -72, upper_bound: -24, segments: 2, critical: false, start_px: 0, end_px: 0, decay_per_ms: 0, total_db: 0, total_px: 0 }
		];
		
		this.reset_levels();
		this.set_peak_freeze(1000);
		this.set_level_decay(1000);
		this.set_level_freeze(50);
		
		for (let section of this.#sections) {
			section.total_db = section.upper_bound - section.lower_bound;
			this.#MIN_LEVEL = section.lower_bound;
		}
	}
	
	static get observedAttributes() {
		return ['style', 'class'];
	}

	attributeChangedCallback(name, old_value, new_value) {
		this.#resize();
	}
	
	disconnectedCallback() {
		if (!this.#parent_element) {
			this.#resize_observer.unobserve(this.#parent_element);
			this.#style_observer.unobserve(document.documentElement);
		}
		this.#app.destroy();
		this.#dom_initialized = false;
	}
	
	connectedCallback() {
		this.style.display = "block";
		this.style.overflow = "clip";
					
		const shadow = this.attachShadow({ mode: 'open' });
		
		shadow.innerHTML = `
			<style>
				:host {
					--font-family: var(--font-family, Sans);
					--font-weight: var(--font-weight, normal);					
					--font-size: var(--font-color, 1em);
					--font-color: var(--font-color, white);
					--gap: var(--padding, 1em);
					--normal-on: var(--normal-on, lightgreen);
					--normal-off: var(--normal-off, darkgreen);
					--critical-on: var(--critical-on, lightcoral);
					--critical-off: var(--critical-off, darkred);
					--scale-color: var(--scale-color, grey);					
				}
				
				#vumeter {
					font-family: var(--font-family);
					font-weight: var(--font-weight);
					font-size: var(--font-size);
				}
				
				#measure {
					position: absolute;
					display: none;
					font-family: var(--font-family);
					font-weight: var(--font-weight);
					font-size: var(--font-size);
					top: var(--gap);
					color: var(--font-color);
					background: var(--scale-color);
				}
			</style>
			
			<div id="measure"></div>
			
			<canvas id="vumeter"></canvas>
		`;
		
		this.#html_measure = shadow.getElementById('measure');
		this.#parent_element = this.parentElement;
		this.#parent_element.style.overflow = 'clip';
	
		const app = new PIXI.Application();	
		app.init({
			view: shadow.getElementById('vumeter'),
			resolution: window.devicePixelRatio,
			autoDensity: true,
			resizeTo: null,
			backgroundAlpha: 0}).then(() => {
				this.#app = app;
				app.ticker.add((ticker) => this.#animate(ticker));
				
				this.#background = new PIXI.Graphics();
				this.#bars = new PIXI.Graphics();
				this.#scale = new PIXI.Graphics();
				this.#fps = new PIXI.Text({text: ""});
				this.#fps.anchor.set(0.5, 0.5);
				const fps_graphics = new PIXI.Graphics();
				fps_graphics.addChild(this.#fps);
				
				app.stage.addChild(this.#background);
				app.stage.addChild(this.#bars);
				app.stage.addChild(this.#scale);
				app.stage.addChild(fps_graphics);
				
				this.#dom_initialized = true;
				
				this.#resize();
				this.#resize_observer.observe(this.parentElement);
				
				this.#style_observer.observe(document.documentElement, { attributes: true, attributeFilter: ['data-bs-theme'] });
			  });
	}
	
	#text(graphic, string, x, y, style) {
		let text = new PIXI.Text({text: string, style: style});
		
		let px = x;
		if (style.text_align === 'center') px -= text.width / 2;
		else if (style.text_align === 'right') px -= text.width;
		
		let py = y;
		if (style.text_baseline === 'alphabetic') py -= style.fontSize;
		else if (style.text_baseline === 'middle') py -= text.height / 2;
		else py -= text.height - style.fontSize;
		
		text.position.set(px, py);
		
		graphic.addChild(text);
	}
	
	#dashed_line(g, x1, y1, x2, y2, dashLength = 4, gapLength = 4) {
	    const dx = x2 - x1;
	    const dy = y2 - y1;
	    const len = Math.sqrt(dx * dx + dy * dy);
	    const angle = Math.atan2(dy, dx);

	    let progress = 0;
	    let x = x1;
	    let y = y1;

	    while (progress < len) {
	        const dashEnd = Math.min(progress + dashLength, len);

	        const xStart = x1 + Math.cos(angle) * progress;
	        const yStart = y1 + Math.sin(angle) * progress;

	        const xEnd = x1 + Math.cos(angle) * dashEnd;
	        const yEnd = y1 + Math.sin(angle) * dashEnd;

	        g.moveTo(xStart, yStart);
	        g.lineTo(xEnd, yEnd);

	        progress = dashEnd + gapLength;
	    }
		g.stroke();
	}
	
	#resize() {
		if (!this.#dom_initialized) return;
		
		let total_segments = 0;
		for (let section of this.#sections) total_segments += section.segments;		
		
		const app = this.#app;
				
		const rect = this.#parent_element.getBoundingClientRect();
		
		const style = window.getComputedStyle(this.#parent_element);
		const padding_top = parseFloat(style.paddingTop) * 1;
		const padding_bottom = parseFloat(style.paddingBottom) * 1;
		const padding_left = parseFloat(style.paddingLeft);
		const padding_right = parseFloat(style.paddingRight);
				
		const width = Math.max(1, rect.width - padding_left - padding_right);
		const height = Math.max(1, rect.height - padding_top - padding_bottom);	
		const dpr = window.devicePixelRatio || 1;
		
		this.#width = width;
		this.#height = height;
		
		const background = this.#background;
		const scale = this.#scale;
		
		background.clear();
		background.removeChildren(); // because of text
		scale.clear();		
		
		// Determine constraints
		const computed_style = getComputedStyle(this.#html_measure);
		const font_family = computed_style.fontFamily;
		const font_style = computed_style.fontStyle;
		const font_weight = computed_style.fontWeight;
		const font_size = parseFloat(computed_style.fontSize);
		const font_size_half = font_size / 2;
		const font_color = computed_style.color;		
		const gap = parseFloat(computed_style.top);
		
		this.#style_normal_on = computed_style.getPropertyValue('--normal-on');
		this.#style_critical_on = computed_style.getPropertyValue('--critical-on');
		const style_normal_off = computed_style.getPropertyValue('--normal-off');
		const style_critical_off = computed_style.getPropertyValue('--critical-off');
		const style_scale_color = computed_style.getPropertyValue('--scale-color');		
		
		const text_style = {
			fill: font_color,
			fontFamily: font_family,
			fontSize: font_size,
			fontWeight: font_weight,
			lineHeight: 1
		};
		
		const measure_text = new PIXI.Text('measure', text_style);
		
		let max_explanation_width = -font_size_half;
		let max_explanation_height = 0;
		this.#meter_height = height;
		let display_channels = true;
		let max_channel_width = 0; // Minimum distances between channels
		if (height > 5 * font_size) {
			for (let channel of this.#channels) {
				measure_text.text = String(channel.name);
				if (max_channel_width < measure_text.width) max_channel_width = measure_text.width;
			}
			const channels_width = 8 * max_channel_width + 7 * font_size_half;
			
			if (channels_width < width) {
				measure_text.text = 'dB';
				if (max_explanation_width < measure_text.width) max_explanation_width = measure_text.width;
				for (let section of this.#sections) {
					measure_text.text = String(section.lower_bound);
					if (max_explanation_width < measure_text.width) max_explanation_width = measure_text.width;
					if (max_explanation_height < measure_text.height) max_explanation_height = measure_text.height; 
				}
				
				if (width - 2 * max_explanation_width - font_size - channels_width < 0) {
					display_channels = false;
				}
			} else display_channels = false;
		} else display_channels = false;
		measure_text.destroy();
		
		if (display_channels) this.#meter_height = height - font_size - gap; 
			
		// Calculate the sections' start and end pixel
		let start_px = 0;
		for (let i = 0; i < this.#sections.length - 1; i++) {
			const section = this.#sections[i];
			section.start_px = start_px;
			section.end_px = start_px + this.#meter_height * section.segments / total_segments;
			section.total_px = section.end_px - start_px;
			start_px = section.end_px; 
		}
		const last_section = this.#sections[this.#sections.length - 1];
		last_section.start_px = start_px;
		last_section.end_px = this.#meter_height;
		last_section.total_px = last_section.end_px - start_px;
		
		// Draw explanation texts
		let explanation_shown = false;
		if (max_explanation_width > -font_size_half) {
			explanation_shown = true;
			text_style.text_align = 'right';
			this.#text(background, '0', max_explanation_width, 0, text_style);
			this.#text(background, '0', width, 0, text_style);
						
			text_style.text_baseline = 'alphabetic';
			this.#text(background, 'dB', max_explanation_width, this.#meter_height, text_style);
			this.#text(background, 'dB', width, this.#meter_height, text_style);
			
			let upper_bound = font_size;
			let lower_bound = this.#meter_height - font_size;
			let explanation_area_height = lower_bound - upper_bound;
			
			const explanation_text_height = max_explanation_height * (total_segments - 1);
			if (explanation_text_height <= explanation_area_height &&  2 * explanation_text_height > explanation_area_height) {
				upper_bound *= 0.75;
				lower_bound = this.#meter_height;
				explanation_area_height = lower_bound - upper_bound;
				const distance = explanation_area_height / (total_segments);
				
				let current_section_idx = 0;
				let current_section = this.#sections[current_section_idx];
				let sum_segments = 0;
				for (let i = 1; i < total_segments; i++) {
					sum_segments++;				
					if (sum_segments >= current_section.segments) {
						sum_segments = 0;
						current_section_idx++;
						current_section = this.#sections[current_section_idx];
					}
					
					upper_bound += distance;
	
					let level = current_section.upper_bound - (current_section.upper_bound - current_section.lower_bound) * sum_segments / current_section.segments;
					this.#text(background, level, max_explanation_width, upper_bound, text_style);
					this.#text(background, level, width, upper_bound, text_style);
				}
			} else {			
				text_style.text_baseline = 'middle';
				let all_labels = true; 
				let current_section_idx = 0;
				let current_section = this.#sections[current_section_idx];
				let sum_segments = 0;
				for (let i = 1; i < total_segments; i++) {
					sum_segments++;				
					if (sum_segments >= current_section.segments) {
						sum_segments = 0;
						current_section_idx++;
						current_section = this.#sections[current_section_idx];
					} else if (!all_labels) continue;
					
					const y = this.#meter_height / total_segments * i;
									
					if (y - font_size_half < upper_bound || y + font_size_half > lower_bound) {
						all_labels = false;
						continue;
					}
					
					upper_bound = y + font_size_half;
	
					let level = current_section.upper_bound - (current_section.upper_bound - current_section.lower_bound) * sum_segments / current_section.segments;
					this.#text(background, level, max_explanation_width, y, text_style);
					this.#text(background, level, width, y, text_style);
				}
			}
		}
		
		// Calculate positions of channel bars
		const meter_start = explanation_shown ? max_explanation_width + font_size_half : 0;
		const meter_width = width - (explanation_shown ? (2 * max_explanation_width + font_size) : 0); // => 2 * 1/2 font_size space to meter left and right
		
		const block_gap = (meter_width / 8) * 0.66;
		const block_width = (meter_width - block_gap * 3) / 4;
		
		const bar_gap = (block_width / 2) * 0.15;
		const bar_width = (block_width - bar_gap) / 2;
		const bar_width_half = bar_width / 2; 		
		this.#bar_thickness = bar_width;
						
		let x = meter_start;
		for (let i = 0; i < 4; i++) {
			this.#channels[i * 2].start_px = x;
			x += bar_width + bar_gap;
			
			this.#channels[i * 2 + 1].start_px = x;
			x += bar_width + block_gap;
		}		
		
		// Draw channel names
		if (display_channels) {
			text_style.text_align = 'center';
			text_style.text_baseline = 'alphabetic';
			for (let channel of this.#channels) {
				this.#text(background, channel.name, channel.start_px + bar_width_half, height, text_style);
			}
		}
		
		// Draw bar background
		const meter_height = this.#meter_height;
		for (let i = 0; i < 8; i++) {
			let x = this.#channels[i].start_px; 
			let y = 0;
			let height = 0;
			
			for (let section of this.#sections) {
				if (!section.critical) {
					height = height;
					background.rect(x, y, bar_width, height).fill({ color: style_critical_off });					
					
					y += height;
					height = meter_height - y;
					background.rect(x, y, bar_width, height).fill({ color: style_normal_off });
					break;
				}
				height += section.end_px - section.start_px;
			}
		}
		
		// Draw scale lines
		scale.setStrokeStyle({ width: 1, color: style_scale_color });
		for (let i = 0; i <= total_segments; i++) {
			this.#dashed_line(scale, 
				max_explanation_width + font_size_half * 0.5, 
				0.5 + (meter_height - 1) / total_segments * i, 
				max_explanation_width + font_size_half * 1.5 + meter_width, 
				0.5 + (meter_height - 1) / total_segments * i);			
		}
		
		this.#fps.position.set(width / 2, height / 2);		
		
		this.#draw_bars();
	}
	
	#draw_bars() {		
		const width = this.#width;
		const height = this.#height;
		const bars = this.#bars;
		const style_normal_on = this.#style_normal_on;
		const style_critical_on = this.#style_critical_on;
		const meter_height = this.#meter_height;
		
		bars.clear();
		
		for (let i = 0; i < 8; i++) {
			let x = this.#channels[i].start_px; 
			let y = 0;
			let height = 0;
			let found = false;
			let level = this.#levels[i];
			
			// Draw levels			
			for (let section of this.#sections) {
				if (section.lower_bound < level && level <= section.upper_bound) {
					const total_section_px = section.total_px;
					const total_section_db = section.total_db;
					const rel_level = section.upper_bound - level;
					
					height = total_section_px * rel_level / total_section_db;					
					y = section.start_px + height;					
					
					if (!section.critical) {
						height = meter_height - y;
						bars.rect(x, y, this.#bar_thickness, height).fill({ color: style_normal_on });
						break;	
					} else found = true;			
				} else if (found && !section.critical) {
					height = section.start_px - y;
					bars.rect(x, y, this.#bar_thickness, height).fill({ color: style_critical_on });
					y = section.start_px;
					height = meter_height - y;
					bars.rect(x, y, this.#bar_thickness, height).fill({ color: style_normal_on }); 
					break;
				}
			}
			
			// Draw peak level
			y = 0;
			level = this.#peak_levels[i];
			if (level > this.#MIN_LEVEL) {
				for (let section of this.#sections) {
					const total_section_px = section.total_px;
					if (section.lower_bound < level && level <= section.upper_bound) {
						const rel_level = section.upper_bound - level;
						
						y = y + total_section_px * rel_level / section.total_db;					
						height = 3;
						if (section.end_px < y + height) y = section.end_px - height; 
						
						const color = section.critical ? style_critical_on : style_normal_on;
						bars.rect(x, y, this.#bar_thickness, height).fill({ color: color });
						break;
					} else {
						y += total_section_px;
					}
				}
			}
		}
	}
	
	#animate(ticker) {
		if (!this.#dom_initialized) return;
		const now = performance.now();
		for (var i = 0; i < 8; i++) {
			// Calc level freeze
			let level = this.#levels[i];
			let skip = this.#levels_ts[i] < this.#level_freeze;
			if (this.#levels_ts[i] < 1000) this.#levels_ts[i] += ticker.deltaMS;
			
			// Calc peak level
			const peak_ts = this.#peak_ts[i];
			if ((level > this.#peak_levels[i]) 
			|| (peak_ts === 0) 
			|| (peak_ts > 0) && (now - peak_ts >= this.#peak_freeze)) {
				this.#peak_levels[i] = level;
				this.#peak_ts[i] = now;				
			}
			
			if (skip) continue;
			
			// Calc level decay
			if (level > this.#MIN_LEVEL) {
				let section_found = false;
				let ts_delta = ticker.deltaMS;
				
				for (let section of this.#sections) {
					if (section_found || (section.lower_bound < level && level <= section.upper_bound)) {
						const decay = section.decay_per_ms * ts_delta;
						if (level - decay >= section.lower_bound) {
							level -= decay;
							ts_delta = 0;
							break;	
						} else {
							section_found = true;
							level = section.lower_bound;
							ts_delta -= Math.max(0, (level - section.lower_bound) / section.decay_per_ms);
						}
					}
				}
				
				if (ts_delta > 0) level = this.#MIN_LEVEL;
				this.#levels[i] = level;
			}
		}
				
		this.#draw_bars();
		
		// Show fps
		if (false) {
			this.#frames_cnt++;
			this.#frames_ts += ticker.deltaMS;
			if (this.#frames_ts >= 1000) {
				const val = Math.round(this.#frames_cnt * 10000 / this.#frames_ts) / 10; 
				this.#frames_ts = 0;
				this.#frames_cnt = 0;
				this.#fps.text = val;
			}	
		}
	}
	
	set_peak_freeze(ms) {
		this.#peak_freeze = ms;
	}
	
	set_level_decay(ms) {
		const decay_per_section = ms / this.#sections.length;
		
		for (let section of this.#sections) {
			const db_range = section.upper_bound - section.lower_bound;
			section.decay_per_ms = db_range / decay_per_section;
		}
	}
	
	set_level_freeze(ms) {
		this.#level_freeze = Math.min(1000, ms);
	}
	
	reset_levels() {
		this.#levels_ts = [];
		this.#levels = [];
		this.#peak_levels = [];
		this.#peak_ts = [];
		for (let i = 0; i < 8; i++) {
			this.#levels.push(this.#MIN_LEVEL);
			this.#peak_levels.push(this.#MIN_LEVEL);
			this.#levels_ts.push(0);
			this.#peak_ts.push(0);
		}		
	}
	
	set_levels(levels) {
		if (!Array.isArray(levels) || levels.length !== 8) return;
				
		for (let i = 0; i < 8; i++) {
			let level = levels[i];
			if (!typeof level === 'number') continue;
			level = Math.max(this.#MIN_LEVEL, Math.min(0, level));
			
			if (level > this.#levels[i]) {
				this.#levels[i] = level;
				this.#levels_ts[i] = 0;
			} 			
		}
	}
	
	start() {
		this.reset_levels();
		
		this.#frames_cnt = 0;
		this.#frames_ts = 0;
		this.#frames_fps = 0;
		
		this.#app.ticker.start();
		console.log("vumeter started");
	}
	
	stop() {
		this.#app.ticker.stop();
		this.reset_levels();
		this.#draw_bars();		
		console.log("vumeter stopped");
	}

}

// Custom element definitions ======================================================================

/**
 * Displays a channel layout matrix and stream informations like format, sample rate or bit depth. 
 * 
 * This component uses following (CSS) styles: width, maxHeight
 * 
 * The component tries to expand to maximal width. The height increases proportional to the width.
 * The maxHeight limits the maximal height and also proportional the maximal width.
 * 
 * The element has following attributes:
 * 
 * title			The caption for the stream info like 'input' or 'output'
 * channel_bitmap	The current channel layout as bit map. If a bit is set, the corresponding channel is displayed as enabled.
 *						Bit 0: FL
 *						Bit 1: FR
 *						Bit 2: C
 *						Bit 3: LFE
 *						Bit 4: BL
 *						Bit 5: BR
 *						Bit 9: SL
 *						Bit 10: SR
 * format			A text like 'PCM', 'AC3' or 'DCA'
 * sample_rate		The sample rate of the signal
 * bits				The bit depth of the signal
 * 
 * You can also access these attributes as property in JS.
 * 
 * The element has following additional CSS attributes:
 * 
 * Settings for the title:
 * --title-font-family: Default: Sans
 * --title-font-weight: Default: normal
 * --title-font-color:	Default: white
 * --title-font-size: 	Default: 3em
 * --title-gap: 		Gap between title and channel matrix. Default: 1.5em
 * 
 * Settings for the channel matrix:
 * --matrix-font-family: 	Default: Sans
 * --matrix-font-weight:	Default normal
 * --matrix-off-bg:			Default: grey
 * --matrix-off-color:		Default: white
 * --matrix-on-bg: 			Default: limegreen
 * --matrix-on-color: 		Default: white
 * 
 * Settings for the signal informations:
 * --signal-font-family:	Sans
 * --signal-font-weight:	normal
 * --signal-font-color:		white
 */
customElements.define('hi-stream-info', StreamInfo);

/**
 * The element <hi-slider> is an alternative web component for <input type='range'/>.
 * 
 * You can set following attributes on the HTML element: min, max, step, value.
 * You can also access these attributes as property in JS.
 * 
 * Subscribe the event 'input' to receive a value change.
 *
 * This component uses following (CSS) styles: width, height
 * 
 * The element has following additional CSS attributes:
 * --color				Sets the color of the indicator. Default: lime 
 * --bg-color			Sets the color of the background. Default: none
 * --height				The height. Default: 100%
 * --width				The width. Default: 100%
 * --orientation		Defines the orientation of the slider:
 *	 					0, 0deg, bt, to top: 		The slider is vertically oriented. 
 * 													The minimal value is at the bottom and maximal value at the top. 
 * 													This is the default.
 * 						90, 90deg, lr, to right: 	The slider is horizontally oriented. 
 * 													The minimal value is at the left and maximal value at the right.
 * 						180, 180deg, tb, to bottom: The slider is vertically oriented. 
 * 													The minimal value is at the top and maximal value at the bottom.
 * 						270, 270deg, rl, to left: 	The slider is horizontally oriented. 
 * 													The minimal value is at the right and maximal value at the left.
 */
customElements.define('hi-slider', Slider);

customElements.define('hi-text', CanvasText);

customElements.define('hi-clock', DigitalClock);

/**
 * The element <hi-carousel> defines a dynamic carousel. You can call following functions:
 * 
 * set_slide(content)			Replaces the current slide with the given content. Returns the element on success otherwise null.
 * 								The content can be a string with html or a js node object.
 * slide_to(content, forward)	Adds a new pane with the content, shows the new pane with a slide animation
 * 								and then removes the old content. Returns the element on success otherwise null.
 * 								The content can be a string with html or a js node object.
 * remove_slides()				Removes all slides.
 */
customElements.define('hi-carousel', Carousel);

/**
 * The element <hi-vumeter> defines a vu meter. 
 * 
 * Requires pixijs V.8.13.2 or higer.
 * 
 * You can call following functions:
 * 
 * start()					Starts the vu meter animation. Requires continues data input by calling set_levels().
 * stop()					Stops the vu meter animation.
 * set_levels(levels)		Sets the current vu meter levels. "levels" is an array of 8 floats as dB values.
 * reset_levels()			Resets the current vu meter levels.
 * 
 * set_peak_freeze(ms)		Sets, how long the peak levels are freezed.
 * set_level_freeze(ms)		Sets, how long the current level is freezed before decay.
 * set_level_decay(ms)		Sets, how fast the current level decays after freeze.
 */
customElements.define('hi-vumeter', VUMeter);