/**
 * Copyright (c) 2024-25 Dieter Steinwedel. All rights reserved.
 *
 * Unauthorized copying and publishing of this file, via any medium is strictly prohibited.
 */

const hostname = window.location.hostname;
const port = window.location.port;

const ws_uri_event = "/ws/events";
const ws_uri_vumeter = "/ws/vumeter";
const rest_uri_control_value = "/control/value";
const rest_uri_client_configuration = "/client/configuration";
const rest_uri_client_notification = "/client/notification";
const rest_uri_system_display = "/system/display";
const rest_uri_application_config = "/application/config";
const http_uri_favicon = "/favicon.ico";
const client_id = "default";

const label_class = "text-primary-emphasis"; 
const control_button_class = "btn-primary";

const control_infos = new Map();

const dlg_overlay = document.getElementById('dlg-overlay');
const dlg_carousel = document.getElementById('dlg-carousel');
const overlay_source = document.getElementById("overlay-source");
const overlay_display = document.getElementById("overlay-display");

const client = document.getElementById("main");

const main_volume_id = "ALSA/Volume/1/0/volume";
const main_volume_mute_id = "ALSA/Volume/1/0/button";
const source_id = "ALSA/Input Capture Source/0/0/list";

const input_status_channel_layout_id = "DSC8/capture/status/channel_layout";
const input_status_format_id = "DSC8/capture/status/format";
const input_status_sample_rate_id = "DSC8/capture/status/sample_rate";
const input_status_bits_id = "DSC8/capture/status/bits";

const output_status_channel_layout_id = "DSC8/playback/status/channel_layout";
const output_status_format_id = "DSC8/playback/status/format";
const output_status_sample_rate_id = "DSC8/playback/status/sample_rate";
const output_status_bits_id = "DSC8/playback/status/bits";

const signal_generator_id = "DSC8/signal_generator";
const signal_generator_output_to_id = signal_generator_id + "/output_to";

const settings_ui_theme_show_dark = "settings/ui/theme/dark_mode";
const settings_ui_clock_show_24h = "settings/ui/clock/show_24h";
const settings_ui_clock_show_seconds = "settings/ui/clock/show_seconds";
const settings_ui_clock_show_date = "settings/ui/clock/show_date";

const settings_ui_display_id = "DSC8/system/display";
const settings_ui_brightness_id = "DSC8/system/display/brightness";

const main_volume_slider = document.getElementById("main-volume");
const main_volume_mute = document.getElementById("main-mute");

const source = document.getElementById("control-source");
const input_info = document.getElementById("control-input-info");
const output_info = document.getElementById("control-output-info");

const main_content_carousel = document.getElementById('main-content-carousel');
const vumeter = document.getElementById("vumeter");
const clock = document.getElementById("clock");

const overlay_volume = document.getElementById('overlay-volume');

let overlay_source_timeout_id;

let button_is_touch = false;
let button_is_down = false;
let button_action_called = false;
let button_timeout_id;
const button_initial_delay = 800;
const button_repeat_delay = 50;

let timeoutId;

let vumeter_enabled = false;

const event_handlers = {
	control_changed_value: event_control_changed_value,
	control_changed_visibility: event_control_changed_visibility,
	control_added: event_control_added,
	control_updated: event_control_updated,
	control_removed: event_control_removed,
	status_info: event_status_info,
	sound_card: event_sound_card,
	client_config_changed: event_client_config_changed,
	client_notification: client_notification,
};
let event_ignore_id = undefined;

let ws_event = null;
let ws_vumeter = null;
let ws_do_reconnect = true;

const settings_menu = {
	id: "settings",
	label: "Settings",
	type: "menu",	
	order: 1000,
	visible: true,
	children: [		
		{			
			id: "settings/remote-access",
			label: "Remote Access",
			type: "menu",
			order: 1000,
			visible: true,
			children: [
				{
					id: "settings/remote-access/android",
					label: "Android",
					type: "menu",
					order: 1000,
					visible: true,
					children: [
						{
							type: "include",
							url: "include/remote-access-android.html"
						}
					]
				},
				{
					id: "settings/remote-access/ios",
					label: "iOS",
					type: "menu",
					order: 2000,
					visible: true,
					children: [
						{
							type: "include",
							url: "include/remote-access-ios.html"
						}
					]
				},
				
				{
					id: "settings/remote-access/browser",
					label: "Web Browser",
					type: "menu",
					order: 3000,
					visible: true,
					children: [
						{
							type: "include",
							url: "include/remote-access-browser.html"
						}
					]
				},
				{
					id: "settings/remote-access/developer",
					label: "Developer",
					type: "menu",
					order: 4000,
					visible: true,
					children: [
						{
							type: "include",
							url: "include/remote-access-developer.html"
						}
					]
				}
			]
		},
		{
			id: "settings/ui",
			label: "User Interface",
			type: "menu",
			order: 2000,
			visible: true,
			children: [
				{
					id: "settings/ui/theme",
					label: "Theme",
					type: "container",
					order: 1000,
					visible: true,
					children: [
						{
							id: settings_ui_theme_show_dark,
							label: "Dark mode",
							type: "toggle",
							setter: set_ui_theme, 
							value: 1,
							min: 0,
							max: 1,
							step: 1,
							order: 1000,
							visible: true,
							value_labels: [
								{
									label: "off",
									value: 0
								},
								{
									label: "on",
									value: 1
								}
							]
						}
					]
				},
				{
					id: settings_ui_display_id,
					type: "ref",
					order: 2000
				},
				{
					id: "settings/ui/clock",
					label: "Clock",
					type: "container",
					order: 3000,
					visible: true,
					children: [
						{
							id: settings_ui_clock_show_24h,
							label: "Show 24h",
							type: "toggle",
							setter: set_clock_show_24h,
							value: 1,
							min: 0,
							max: 1,
							step: 1,
							order: 1000,
							visible: true,
							value_labels: [
								{
									label: "off",
									value: 0
								},
								{
									label: "on",
									value: 1
								}
							]
						},
						{
							id: settings_ui_clock_show_seconds,
							label: "Show seconds",
							type: "toggle",
							setter: set_clock_show_seconds,
							value: 0,
							min: 0,
							max: 1,
							step: 1,
							order: 2000,
							visible: true,
							value_labels: [
								{
									label: "off",
									value: 0
								},
								{
									label: "on",
									value: 1
								}
							]
						},
						{
							id: settings_ui_clock_show_date,
							label: "Show date",
							type: "toggle",
							setter: set_clock_show_date,
							value: 1,
							min: 0,
							max: 1,
							step: 1,
							order: 3000,
							visible: true,
							value_labels: [
								{
									label: "off",
									value: 0
								},
								{
									label: "on",
									value: 1
								}
							]
						}
					]
				}
			]
		},
		{			
			id: "settings/output",
			label: "Output",
			type: "menu",
			order: 3000,
			visible: true,
			children: [
				{			
					id: "settings/output/channel_volume",
					label: "Channel Volume",
					type: "menu",
					order: 1000,
					visible: true,
					children: [
						{
							id: "ALSA/Volume/0",
							type: "children-ref",
							order: 1000
						},
						{
							id: "DSC8/signal_generator",
							type: "ref",
							order: 2000
						}
					]
				},
				{			
					id: "settings/output/channel_mapping",
					label: "Channel Mapping",
					type: "menu",
					order: 2000,
					visible: true,
					children: [
						{
							id: "DSC8/playback/channel_mapping",
							type: "children-ref",
							order: 1000
						},
						{
							id: "DSC8/signal_generator",
							type: "ref",
							order: 2000
						}
					]
				},
				{
					id: "settings/output/phase_inverter",
					label: "Phase Inverter",
					type: "menu",
					order: 3000,
					visible: true,
					children: [
						{
							id: "DSC8/playback/channel_phase",
							type: "children-ref",
							order: 1000
						},
						{
							id: "DSC8/signal_generator",
							type: "ref",
							order: 2000
						}
					]
				},
				{
					id: "settings/output/active_outputs",
					label: "Active Outputs",
					type: "menu",
					order: 4000,
					visible: true,
					children: [
						{
							id: "ALSA/active/outputs",
							type: "children-ref",
							order: 1000
						}
					]
				}
			]
		},
		{
			id: "settings/signal_generator",
			label: "Test Signal",
			type: "menu",
			order: 4000,
			visible: true,
			children: [
				{
					id: "DSC8/signal_generator",
					type: "children-ref",
					order: 1000
				}
			]
		},
		{
			id: "settings/informations",
			label: "Informations",
			type: "menu",
			order: 5000,
			visible: true,
			children: [
				{
					id: "settings/informations/system",
					label: "System",
					type: "menu",
					order: 1000,
					visible: true,
					children: [
						{
							type: "include",
							url: "include/info-system.html"
						}
					]
				},
				{
					id: "settings/informations/application",
					label: "Audio Hardware",
					type: "menu",
					order: 2000,
					visible: true,
					children: [
						{
							type: "include",
							url: "include/info-audio-hw.html"
						}
					]
				}
			]
		}
	]
};

const settings_menu_map = new Map();

let configuration_loaded = false;

let overlay_msg_connecting = null;
let overlay_msg_sound_card_not_connected = null;
let overlay_msg_not_full_featured_mode = null;
let overlay_msg_capture_device_in_use = null;
let overlay_msg_capture_device_has_a_problem = null;
let overlay_msg_playback_device_in_use = null;
let overlay_msg_playback_device_has_a_problem = null;

// =====================================================================

function to_ws_url(uri) {
	return "ws://" + hostname + ":" + port + uri;
}

function to_http_url(uri) {
	return "http://" + hostname + ":" + port + uri;
}

function to_item_id(id) {
	return "frame_" + id;
}

function to_value_id(id, value) {
	return id + "-" + value;
}

function to_string_length(value) {
	let str = value.toString();
	return str.length;
}

function sleep(ms) {
	return new Promise(resolve => setTimeout(resolve, ms));
}

function create_qrcode(loading_elem_id, error_elem_id, no_network_elem_id, content_elem_id, url_elem_id, qrcode_elem_id, path = "/") {
	const rest_uri_application_config = "/system/info";
	const loading_elem = document.getElementById(loading_elem_id);
	const error_elem = document.getElementById(error_elem_id);
	const no_network_elem = document.getElementById(no_network_elem_id);
	const content_elem = document.getElementById(content_elem_id);
	const url_elem = document.getElementById(url_elem_id);
	const qrcode_elem = document.getElementById(qrcode_elem_id);

	fetch(rest_uri_application_config)
	.then(response => {
		if (!response.ok) throw new Error('Network response was not ok');
		return response.json();
	})
	.then(system_info => {
		if (!Array.isArray(system_info.fqdn)) throw new Error('Cannot retrieve a fqdn.');
		
		if (system_info.fqdn.length === 0) {
			loading_elem.classList.add('d-none');
			no_network_elem.classList.remove('d-none');
			return;
		}
	
		const protocol = document.location.protocol + '//';
		const port = (document.location.port === '' || document.location.port === '80' || document.location.port === '443') ? "" : ":" + document.location.port;
		const url = protocol+ system_info.fqdn[0] + port + path;
		url_elem.textContent = url;
		const style = getComputedStyle(document.body);
		
		const dpr = window.devicePixelRatio || 1;
		const size = Math.min(window.innerWidth, window.innerHeight) * dpr / 3;
		
		const bg_color = style.color;		
		const color = style.backgroundColor;
		
		new QRCode(qrcode_elem, {
			text: url,
			width: size,
			height: size,
			colorDark: bg_color,
			colorLight: color,
			correctLevel: QRCode.CorrectLevel.H
		});

		loading_elem.classList.add('d-none');
		content_elem.classList.remove('d-none');
	}).catch(error => {
		console.error('Cannot get system info:', error);
		loading_elem.classList.add('d-none');
		error_elem.classList.remove('d-none');
	});
}

function set_display(new_state) {
	if (!overlay_display) return;
	
	let is_off = overlay_display.classList.contains('off');
	
	switch (new_state) {
		case 'on':
			if (!is_off) return;
			
			fetch(rest_uri_system_display + "?state=on", {
				method: 'POST',
				headers: { 'Accept': '*/*', 'Content-Type': 'application/json' }
			}).finally(() => {
				setTimeout(() => {
					overlay_display.classList.remove('off');
				}, 300);
			});			
			
			break;
			
		case 'off':
			if (is_off) return;							
			fetch(rest_uri_system_display + "?state=off", {
				method: 'POST',
				headers: { 'Accept': '*/*', 'Content-Type': 'application/json' }
			});	
			overlay_display.classList.add('off');
			break;
			
		case 'toggle':
			set_display(is_off ? 'on' : 'off');
			break;
	}
}

function change_theme_color(color) {
    let themeColorMeta = document.querySelector('meta[name="theme-color"]');
    if (themeColorMeta) {
        themeColorMeta.setAttribute('content', color);
    } else {
        themeColorMeta = document.createElement('meta');
        themeColorMeta.name = 'theme-color';
        themeColorMeta.content = color;
        document.head.appendChild(themeColorMeta);
    }
}

async function wait_until_reachable() {
	console.log("Testing for server.");
	let test_url = to_http_url(http_uri_favicon);
	while (true) {
		let controller = new AbortController();
		let controller_timeout = setTimeout(() => controller.abort(), 1000);			
		try {	
			let response = await fetch(test_url, { signal: controller.signal });
			if (response.status != 200) throw new Error();
			console.log("Server is reachable.");
			break;
		} catch(error) {
			// Connection timed out; let's retry ...
			await sleep(100);
		}
	}
}

function create_caption(caption) {
	let result = document.createElement('div');
	result.className = 'caption';
	result.innerHTML = caption;
	return result;
}

function add_button_listeners(button, control_id, inc) {
	
	function change_value() {
		let control_info = control_infos.get(control_id);
		if (!control_info) return;
		
		button_action_called = true;
		
		if (inc) {
			if (control_info.value == control_info.max) return;		
			control_info.value += control_info.step;
			if (control_info.value > control_info.max) control_info.value = control_info.max;
		} else {
			if (control_info.value == control_info.min) return;
			control_info.value -= control_info.step;
			if (control_info.value < control_info.min) control_info.value = control_info.min;	
		}
		
		for (const listener of control_info.change_listeners) listener(control_info.value, true);
		
		send_control_value(control_info);
	}
	
	function button_action() {
		change_value();
		button_timeout_id = setTimeout(button_action, button_repeat_delay);
	}

	function button_down() {
		if (button_is_down) return;
		button_is_down = true;
		button_action_called = false;
		event_ignore_id = control_id;
		button_timeout_id = setTimeout(button_action, button_initial_delay);
	}
	
	function button_up() {
		if (!button_is_down) return;
		
		clearTimeout(button_timeout_id);
		
		event_ignore_id = undefined;		
		button_is_down = false;
		
		if (!button_action_called) {
			change_value();
			button_action_called = false;
		}
	}
	
	button.addEventListener('touchstart', function(e) {
		e.preventDefault();
		button_is_touch = true;
		button_down();
	});
	
	button.addEventListener('touchend', function(e) {
		e.preventDefault();
		button_up();
	});
	
	document.addEventListener('touchmove', function(e) {
		if (!e.cancelable) {
			clearTimeout(button_timeout_id);
			button_is_touch = false;
			button_is_down = false;
			return; 
		}
		e.preventDefault();
	});
	
	button.addEventListener('mousedown', function() {
		if (button_is_touch) return;
		button_down();
	});
	
	button.addEventListener('mouseup', function() {
		if (button_is_touch) return;
		button_up();
	});
	
	button.addEventListener('mouseleave', function() {
		if (button_is_touch) return;
		button_up();
	});
	
	button.addEventListener('keydown', function(event) {
		if (event.code === 'Enter' || event.code === 'Space') button_down();				
	});
	
	button.addEventListener('keyup', function(event) {
		if (event.code === 'Enter' || event.code === 'Space') button_up();
	});
}

function create_value_button(control_info, label, inc, text) {
	let button = document.createElement("button");
	button.setAttribute("class", "btn " + control_button_class);
	button.innerHTML = label;	
	
	add_button_listeners(button, control_info.id, inc);
	
	return button;
}

function is_bit_set(value, position) {
    let mask = 1 << position;
    return (value & mask) !== 0;
}

function get_short_channel_name(position) {
	switch (position) {
		case 0: return "FL";
		case 1: return "FR";
		case 2: return "FC";
		case 3: return "LFE";
		case 4: return "BL";
		case 5: return "BR";
		case 9: return "SL";
		case 10: return "SR";
		default: return "";
	}
}

function auto_space(dst, src) {
	if (dst === "") return src;
    else return dst + " " + src;
}

function get_translated_value(control_info, value) {
	for (let i = 0; i < control_info.value_labels.length; i++) {
		let value_label = control_info.value_labels[i];
		if (value_label.value === value) {
			return value_label.label;
		}
	}
	return value;
}

function log_css_variables() {
	const debug_div = document.createElement('div');
    debug_div.style.position = 'absolute';
    document.body.appendChild(debug_div);
	
	const rootStyles = getComputedStyle(document.documentElement);

	for (let i = 0; i < rootStyles.length; i++) {
	    const propertyName = rootStyles[i];
	    if (propertyName.startsWith('--') && !propertyName.startsWith('--bs') && !propertyName.startsWith('--hi')) {
			debug_div.style.maxHeight = rootStyles.getPropertyValue(propertyName);
			const cs = getComputedStyle(debug_div);
	        console.log(`${propertyName}: ${cs.maxHeight}`);
	    }
	}	
}

function hide_volume_overlay() {
	clearTimeout(timeoutId);
    timeoutId = setTimeout(() => {
        overlay_volume.classList.remove('show');
    }, 750);
}

// Message queue implementation ========================================

const message_queue = [];
let is_sending = false;

function send_control_value(control_info) {
	if (control_info.hasOwnProperty("setter")) {
		let configuration = {};
		
		settings_menu_map.forEach((control_info, key) => {
			if (control_info.hasOwnProperty("setter")) {
				const id = control_info.id.replace(/\//g, "_");
				configuration[id] = control_info.value;
			}
		});
				
		fetch(rest_uri_client_configuration + "?id=" + client_id, {
			method: 'PUT',
			headers: { 'Accept': '*/*', 'Content-Type': 'application/json' },
			body: JSON.stringify(configuration)
		});
		
		return;
	}
	
	const message = {
		control_changed_value: {
			id: control_info.id,
			value: control_info.value
		}
	};
	enqueue_message(message); // Ensure the correct message order
}

function send_message(message) {
	return new Promise((resolve) => {
		if (ws_event) ws_event.send(JSON.stringify(message));
		resolve();
	});
}

function process_queue() {
	if (message_queue.length === 0 || is_sending) {
		return;
	}

	is_sending = true;
	const message = message_queue.shift();

	send_message(message).then(() => {
		is_sending = false;
		process_queue();
	});
}

function enqueue_message(message) {
	const existing_index = message_queue.findIndex(msg => msg.type === "value_changed" && msg.data.id === message.data.id);
	if (existing_index !== -1) message_queue[existing_index] = message;
	else message_queue.push(message);
	process_queue();
}

// Create control functions ============================================

function is_wrapped(elem, ref_top) {
	if (!elem) return false;
	const rect = elem.getBoundingClientRect();				
	return ref_top <= rect.top;
}

function get_wrapped_info(parent, cls, ref_top) {
	const element = parent.querySelector(cls);
	return {
		elem: element,
		is_wrapped: is_wrapped(element, ref_top),
		has_css: element ? element.classList.contains("wrapped") : false
	}	
}

const control_block_observer = new ResizeObserver(entries => {
	for (let entry of entries) {
		const block = entry.target;
		const caption_rect = block.querySelector(".caption").getBoundingClientRect();
		const ref_top = caption_rect.top + caption_rect.height;
		
		const control = get_wrapped_info(block, '.control', ref_top);
		const value_unit = get_wrapped_info(block, '.value_unit', ref_top);
		const aux = get_wrapped_info(block, '.aux', ref_top);
		
		if (control.has_css) {
			if (!aux.is_wrapped && !control.is_wrapped && !value_unit.is_wrapped) {
				control.elem.classList.remove("wrapped");
				continue;
			}			
		} else {
			if (aux.is_wrapped || control.is_wrapped || value_unit.is_wrapped) {
				control.elem.classList.add("wrapped");
				continue;
			}			
		}
		if (aux.is_wrapped != aux.has_css) {
			if (aux.is_wrapped) aux.elem.classList.add("wrapped");
			else aux.elem.classList.remove("wrapped");
			continue;
		}
		if (value_unit.is_wrapped != value_unit.has_css) {
			if (value_unit.is_wrapped) value_unit.elem.classList.remove("wrapped");
			else value_unit.elem.classList.add("wrapped");
			continue;
		}
	}
});

function add_alternative_layout_observer(block) {
	control_block_observer.observe(block);
}

function register_setter(control_info) {
	if (control_info.hasOwnProperty("setter")) {
		control_info.change_listeners.push(function(value, remote) {
			control_info.setter();
		});
	}
}

function create_container(control_info) {
	let nodes = [];
	
	for (let child of control_info.children) nodes.push(create_control(child));
	
	return dlg_create_group(control_info.label, nodes);
}

function create_value_unit(control_info) {
	let value_unit = document.createElement('div');
	value_unit.setAttribute("class", 'value_unit');
	
	let value = document.createElement("div");
	value.className = "value";
	let max_length = Math.max(
		to_string_length(control_info.min), 
		to_string_length(control_info.max),
		to_string_length(control_info.min + control_info.step), 
		to_string_length(control_info.max - control_info.step));
	value.setAttribute("style", "width: " + max_length + "ch;");
	value_unit.appendChild(value);
	
	if (control_info.unit) {
		let unit = document.createElement("div");
		unit.className = "unit";
		unit.textContent = control_info.unit;
		value_unit.appendChild(unit);
	}
		
	control_info.change_listeners.push(function(val) {
		value.textContent = get_translated_value(control_info, val);
	});
	
	return value_unit;
}

function create_simple_label(label, value) {
	let result = document.createElement('div');
	result.setAttribute('class', 'dlg-group-item');
	add_alternative_layout_observer(result);
			
	result.appendChild(create_caption(label));
	
	let value_elem = document.createElement("div");
	value_elem.className = "control label";
	value_elem.textContent = value;
	
	result.appendChild(value_elem);
	
	return result;
}

function create_label(control_info) {
	let result = document.createElement('div');
	result.setAttribute('class', 'dlg-group-item');
	result.setAttribute('id', to_item_id(control_info.id));
	add_alternative_layout_observer(result);
			
	result.appendChild(create_caption(control_info.label));
	
	result.appendChild(create_value_unit(control_info));
	
	return result;
}

function combobox_set_value(control, value) {
	const active = control.querySelectorAll('.active');
	active.forEach(element => element.classList.remove('active'));
	
	const placeholder = control.querySelectorAll('.dropdown-placeholder');
	placeholder.forEach(element => {
		if (element.dataset.value == value) element.classList.add('active');
	});
	
	const dropdown = control.querySelectorAll('.dropdown-item');
	dropdown.forEach(element => {
		if (element.dataset.value == value) element.classList.add('active');
	});
}

function cb_list_on_changed(control_info, control, value) {
	combobox_set_value(control, value);
	
	control_info.value = Number(value);
	
	if ('change_listeners' in control_info) {
		for (const listener of control_info.change_listeners) listener(control_info.value, false);
	}
				
	send_control_value(control_info);
}

function create_list(control_info) {
	let result;
	let block;
	
	block = document.createElement("div");
	block.setAttribute("class", "dlg-group-item");
	block.appendChild(create_caption(control_info.label));
	result = block;
		
	result.setAttribute('id', to_item_id(control_info.id));
	add_alternative_layout_observer(block);
	
	let control = document.createElement("div");
	control.setAttribute("class", "control");
	block.appendChild(control);
	
	// Create button of combobox
	let select_button = document.createElement("button");
	select_button.setAttribute("id", control_info.id);
	select_button.setAttribute("class", "btn btn-outline-secondary dropdown-toggle");
	select_button.setAttribute("type", "button");		
	select_button.setAttribute("data-bs-toggle", "dropdown");
	select_button.setAttribute("aria-label", control_info.label);
	select_button.setAttribute("aria-expanded", "false");		
	control.appendChild(select_button);
	
	// Button placeholder
	let select_placeholder = document.createElement("span");
	select_button.appendChild(select_placeholder);
	
	// Dropdown menu
	let dropdown = document.createElement("ul");
	dropdown.setAttribute("class", "dropdown-menu dropdown-scrollable");
	dropdown.setAttribute("aria-labelledby", control_info.id);
	control.appendChild(dropdown);
			
	// Add items to placeholder and dropdown menu
	for (let value = control_info.min; value <= control_info.max; value += control_info.step) {
		let placeholder_item = document.createElement("div");
		placeholder_item.setAttribute("data-value", value);
		select_placeholder.appendChild(placeholder_item);
		
		let dropdown_item = document.createElement("li");
		dropdown_item.setAttribute("data-value", value);
		dropdown.appendChild(dropdown_item);			
					
		if (value === control_info.value) {
			placeholder_item.setAttribute("class", "dropdown-placeholder active");
			dropdown_item.setAttribute("class", "dropdown-item active");
		} else {
			placeholder_item.setAttribute("class", "dropdown-placeholder");
			dropdown_item.setAttribute("class", "dropdown-item");
		}
		
		let label = get_translated_value(control_info, value);
		if ('unit' in control_info) label = label + control_info.unit;
		 
		placeholder_item.textContent = label;
		dropdown_item.textContent = label;
		
		dropdown_item.addEventListener('mousedown', event => cb_list_on_changed(control_info, control, value));
		dropdown_item.addEventListener('touchstart', event => cb_list_on_changed(control_info, control, value));
	}
	
	if ('change_listeners' in control_info) {
		control_info.change_listeners.push(function(value, remote) {
			if (remote) cb_list_on_changed(control_info, control, value);
		});
	}
	
	register_setter(control_info);
	
	let control_aux = document.createElement("div");
	control_aux.className = "aux btn-group";
	control_aux.setAttribute("role", "group");
	block.appendChild(control_aux);
	control_aux.appendChild(create_value_button(control_info, '<i class="bi bi-caret-down-fill"></i>', false));
	control_aux.appendChild(create_value_button(control_info, '<i class="bi bi-caret-up-fill"></i>', true));	
	
	return result;	
}

function create_toggle(control_info) {
	let result = document.createElement('div');
	result.setAttribute('class', 'dlg-group-item');
	result.setAttribute('id', to_item_id(control_info.id));
	add_alternative_layout_observer(result);
	
	result.appendChild(create_caption(control_info.label));
	
	let control = document.createElement('div');
	control.setAttribute("class", "control form-switch");
	result.appendChild(control);
	
	let toggle = document.createElement("input");
	toggle.setAttribute("id", control_info.id);
	toggle.setAttribute("class", "form-check-input");
	toggle.setAttribute("type", "checkbox");
	control.appendChild(toggle);
	
	toggle.addEventListener("change", function() {
		let value = toggle.checked ? 1 : 0;
		control_info.value = value;
				
		for (const listener of control_info.change_listeners) listener(value, false);
		
		send_control_value(control_info);
	});
	
	control_info.change_listeners.push(function(value, remote) {
		if (remote) toggle.checked = value === 1 ? true : false;
	});
	
	register_setter(control_info);

	return result;	
}

function add_slider_listeners(slider, control_id) {
	slider.addEventListener('mousedown', function() {
		event_ignore_id = control_id;
	});
	
	slider.addEventListener('mouseup', function() {
		let control_info = control_infos.get(control_id);
		event_ignore_id = undefined;
	});
	
	slider.addEventListener("input", function(e) {
		let value = parseFloat(slider.value);
		
		let control_info = control_infos.get(control_id);
		if (control_info) control_info.value = value;
		
		for (const listener of control_info.change_listeners) listener(value, false);
		
		if (control_info) send_control_value(control_info);
	});
}

function create_slider(control_info) {
	const result = document.createElement('div');
	result.className = 'dlg-group-item';	
	result.setAttribute('id', to_item_id(control_info.id));
	add_alternative_layout_observer(result);
			
	result.appendChild(create_caption(control_info.label));
	
	const value = create_value_unit(control_info);
	result.appendChild(value);
	
	const control = document.createElement("div");
	control.setAttribute('class', 'control fill');
	result.appendChild(control);
	
	const slider = document.createElement("hi-slider");
	slider.setAttribute("id", control_info.id);
	slider.setAttribute("type", "range");
	slider.setAttribute("min", control_info.min);
	slider.setAttribute("max",  control_info.max);
	slider.setAttribute("step",  control_info.step);
	control.appendChild(slider);
	
	add_slider_listeners(slider, control_info.id);
	
	register_setter(control_info);
	
	const control_aux = document.createElement("div");
	control_aux.className = "aux btn-group";
	control_aux.setAttribute("role", "group");
	result.appendChild(control_aux);
	
	control_info.change_listeners.push(function(val, remote) {
		if (remote)	slider.value = val;
	});
		
	control_aux.appendChild(create_value_button(control_info, '<i class="bi bi-caret-down-fill"></i>', false, value));
	control_aux.appendChild(create_value_button(control_info, '<i class="bi bi-caret-up-fill"></i>', true, value));
					
	return result;	
}

function create_channel_layout(control_info) {
	let result = document.createElement('div');
	result.className = 'dlg-group-item';
	result.setAttribute('id', to_item_id(control_info.id));
	add_alternative_layout_observer(result);
			
	result.appendChild(create_caption(control_info.label));
	
	let value = document.createElement('div');	
	value.setAttribute('id', control_info.id);
	value.className = 'value_unit';
	result.appendChild(value);
	
	control_info.change_listeners.push(function(val, remote) {
		let layout = "";
		if (is_bit_set(val, 0)) layout = auto_space(layout, get_short_channel_name(0));
		if (is_bit_set(val, 1)) layout = auto_space(layout, get_short_channel_name(1));
		if (is_bit_set(val, 2)) layout = auto_space(layout, get_short_channel_name(2));
		if (is_bit_set(val, 3)) layout = auto_space(layout, get_short_channel_name(3));
		if (is_bit_set(val, 9)) layout = auto_space(layout, get_short_channel_name(9));
		if (is_bit_set(val, 10)) layout = auto_space(layout, get_short_channel_name(10));
		if (is_bit_set(val, 4)) layout = auto_space(layout, get_short_channel_name(4));
		if (is_bit_set(val, 5)) layout = auto_space(layout, get_short_channel_name(5));
		if (layout === "") layout = "No channel active";
		value.innerHTML = layout;
	});
	
	return result;	
}

function create_menu_item(control_info) {
	let result = document.createElement('div');
	result.className = "dlg-menu-item dlg-group-item";
	result.setAttribute('id', to_item_id(control_info.id));
	result.setAttribute('onclick',"slide_to_settings_menu('" + control_info.id + "', true, '" + to_item_id(control_info.id) + "')");
	
	const label = document.createElement('span');
	label.textContent = control_info.label;
	result.appendChild(label);
	
	const chevron = document.createElement('i');
	chevron.className = 'bi bi-chevron-right dlg-menu-item-link';
	result.appendChild(chevron);
		
	return result;
}

function create_control(control_info) {
	let control;
			
	switch (control_info.type) {
		case "include":
			control_info.visible = true;
			control = document.createElement('div');
			control.textContent = 'Loading ...';
			
			fetch(control_info.url)
			.then(response => {
				if (!response.ok) throw new Error('Network response was not ok');
				return response.text();
			})				
			.then(data => {
				control.textContent = '';
				const parser = new DOMParser();
				const doc = parser.parseFromString(data, 'text/html');
				while (doc.body.firstChild) control.appendChild(doc.body.firstChild);
				
				const scripts = control.getElementsByTagName('script');
				for (let script of scripts) {
					const temp_fn = new Function(script.textContent);
					temp_fn();
		        }
			}).catch(error => console.error('Cannot include html:', error));
			
			break;
		
		case "ref":			
			const ref = control_infos.get(control_info.id);
			control_info.visible = ref.visible;
			control = create_control(ref);			
			break;
			
		case "children-ref":
			const children_ref = control_infos.get(control_info.id);
			if (children_ref) {
				if (children_ref.children.length > 0 && children_ref.children[0].type !== "container") {
					let items = [];
					for (const ref_child of children_ref.children) {
						items.push(create_control(ref_child));
					}
					control = dlg_create_group(null, items);
				} else {
					control = document.createElement('div');
					for (const ref_child of children_ref.children) {
						control.appendChild(create_control(ref_child));
					}
				}
				control_info.visible = children_ref.visible;
			}
			break;
			
		case "menu":
			control_info.visible = true;
			control = create_menu_item(control_info);
			break;
		
		case "container":			
			control = create_container(control_info);	
			break;
			
		case "list":
			control = create_list(control_info);			
			break;
			
		case "toggle":
			control = create_toggle(control_info);
			break;
			
		case "slider":
			control = create_slider(control_info);
			break;
			
		case "channel_layout":
			control = create_channel_layout(control_info);			
			break;
			
		case "label":
			control = create_label(control_info);
			break;
			
		default:
			console.log("Implementation missing for event 'create control':\n" + JSON.stringify(control_info));
			break;
	}
	
	if (control_info.hasOwnProperty("change_listeners")) {
		for (const listener of control_info.change_listeners) listener(control_info.value, true);
	}
	
	if (control != null) {
		if (control_info.visible) {
			if (control.classList) control.classList.remove('d-none');
		} else {
			if (control.classList) control.classList.add('d-none');
			else control.setAttribute('class', 'd-none');
		}
	}

	return control;
}

function link_main_control(control_info) {
	if (control_info.type === "container") {
		for (let child of control_info.children) link_main_control(child);
		return;
	}
	
	switch (control_info.id) {
		case main_volume_id:
			let omit = true;
			
			let cb_volume = function(value, remote) {
				if (!remote || (remote && !main_volume_slider.is_sliding)) {
					overlay_volume.textContent = String(value) + " dB";
			        if (!omit) overlay_volume.classList.add('show');
				}
				
				if (remote && !main_volume_slider.is_sliding) {
					if (main_volume_slider) main_volume_slider.value = value; 						
			        hide_volume_overlay();
				}
				
				omit = false;
			};
			
			control_info.change_listeners.push(cb_volume);
			
			if (main_volume_slider) main_volume_slider.value = control_info.value;
			
			break;
			
		case main_volume_mute_id:
			let cb_mute = function(value, remote) {
				if (remote) control_info.value = value;
				
				if (main_volume_mute) {				
					if (value) main_volume_mute.classList.add("active");
					else main_volume_mute.classList.remove("active");
				}
				if (main_volume_slider) {
					if (value) main_volume_slider.classList.add("muted");
					else main_volume_slider.classList.remove("muted");
				}
			};
			
			control_info.change_listeners.push(cb_mute);
			
			cb_mute(control_info.value, false);
			
			break;
			
		case source_id:
			const id_prefix = 'main-source-';	

			let new_source = get_translated_value(control_info, control_info.value);
			if (source) source.textContent = new_source;
			
			control_info.change_listeners.push(function(value) {
				// Update overlay-source-list
				let new_source = get_translated_value(control_info, value);
				if (source) source.textContent = new_source;
				
				// Update source dlg if opened
				update_source_list(value);
				
				// Update overlay-source
				const items = main_content_carousel.querySelectorAll('.carousel-item');
			    let current_index = 0;

			    items.forEach((item, index) => {
			        if (item.classList.contains('active')) {
			            current_index = index;
			        }
			    });
				
				if (current_index !== 0) {
					overlay_source.innerHTML = new_source;
					overlay_source.style.visibility = "visible";
					overlay_source.style.opacity = "1";
					
					if (overlay_source_timeout_id) {
				        clearTimeout(overlay_source_timeout_id);
				    }
					
					overlay_source_timeout_id = setTimeout(function() {
					    overlay_source.style.visibility = "hidden";
					    overlay_source.style.opacity = "0";
					}, 1000);
				}
			});
			
			//TODO update control value
			
			break;
			
		case input_status_channel_layout_id:
			control_info.change_listeners.push(function(value, remote) {
				if (input_info) input_info.channel_bitmap = value;
			});
			if (input_info) input_info.channel_bitmap = control_info.value;
			break;
			
		case output_status_channel_layout_id:
			control_info.change_listeners.push(function(value, remote) {
				if (output_info) output_info.channel_bitmap = value;
			});
			if (output_info) output_info.channel_bitmap = control_info.value;
			break;
		
		case input_status_format_id:
			control_info.change_listeners.push(function(value, remote) {
				if (input_info) input_info.format = get_translated_value(control_info, value);
			});
			if (input_info) input_info.format = get_translated_value(control_info, control_info.value);
			break;
			
		case output_status_format_id:
			control_info.change_listeners.push(function(value, remote) {
				if (output_info) output_info.format = get_translated_value(control_info, value);
			});
			if (output_info) output_info.format = get_translated_value(control_info, control_info.value);
			break;
			
		case input_status_sample_rate_id:	
			control_info.change_listeners.push(function(value, remote) {
				if (input_info) input_info.sample_rate = value;
			});
			if (input_info) input_info.sample_rate = control_info.value;
			break;
			
		case output_status_sample_rate_id:
			control_info.change_listeners.push(function(value, remote) {
				if (output_info) output_info.sample_rate = value;
			});
			if (output_info) output_info.sample_rate = control_info.value;
			break;
			
		case input_status_bits_id:
			control_info.change_listeners.push(function(value, remote) {
				if (input_info) input_info.bits = value;
			});
			if (input_info) input_info.bits = control_info.value;
		break;
		
		case output_status_bits_id:
			control_info.change_listeners.push(function(value, remote) {
				if (output_info) output_info.bits = value;
			});
			if (output_info) output_info.bits = control_info.value;
			break;
	}
}

function update_control_view() {
	const sg_control_info = control_infos.get(signal_generator_output_to_id);
	if (!sg_control_info) return;
	
	let signal_generator_enabled = false;
	if ('children' in sg_control_info) {
		sg_control_info.children.forEach(child => {
			if (child.value === 1) signal_generator_enabled = true;
		});
	}
	
	const source_control_info = control_infos.get(source_id);
	if (source_control_info && source) {
		if (signal_generator_enabled) {
			source.textContent = 'Test Signal';
			input_info.classList.add('disabled');
		} else {
			source.textContent = get_translated_value(source_control_info, source_control_info.value);
			input_info.classList.remove('disabled');
		}
	}
}

function link_signal_generator(control_info) {
	if (!control_info || !control_info.id.startsWith(signal_generator_id)) return;
	
	if (control_info.id === signal_generator_output_to_id) {		
		const listener = (value, remote) => {
			update_control_view();
		};
		
		if ('children' in control_info) {
			control_info.children.forEach(child => child.change_listeners.push(listener));
		}
	} else {
		if ('children' in control_info) {
			control_info.children.forEach(child => link_signal_generator(child));
		}
	}
}

// Event functions =====================================================

function add_to_control_infos(control_info) {
	control_infos.set(control_info.id, control_info);
	control_info.change_listeners = [];
	
	for (let child of control_info.children) add_to_control_infos(child);
}

function event_control_added(controls) {
	for (let control_info of controls) {
		add_to_control_infos(control_info);
		if (control_info.category !== "settings") link_main_control(control_info);
		else link_signal_generator(control_info);	
	}
	update_control_view();
}

function event_control_updated(controls) {
	let ids = [];
	for (let control_info of controls) {
		ids.push(control_info.id);
		if (control_info.children) event_control_updated(control_info.children);
	}
	event_control_removed(ids);
	event_control_added(controls);
}

function event_control_removed(controls) {
	for (let control_id of controls) {
		control_infos.delete(control_id);
		let div = document.getElementById(to_item_id(control_id));
		if (div) div.remove();
	}
}

function event_control_changed_value(changes) {
	for (let changed of changes) {
				
		if (event_ignore_id === changed.id) continue;
		
		let control_info = control_infos.get(changed.id);
		if (!control_info) {
			if (ws_event) ws_event.close();
			return;
		}

		if (control_info.value === changed.value) continue;
		control_info.value = changed.value;
				
		for (const listener of control_info.change_listeners) listener(changed.value, true);
	}
}

function event_control_changed_visibility(changes) {
	for (let changed of changes) {
		let control_info = control_infos.get(changed.id);
		if (!control_info) {
			if (ws_event) ws_event.close();
			return;
		}
		
		if (control_info.visible === changed.visible) continue;
		
		control_info.visible = changed.visible;
		
		let element = document.getElementById(to_item_id(changed.id));
		if (!element) {
			if (ws_event) ws_event.close();
			return;
		}
		
		if (control_info.visible) element.classList.remove('d-none');
		else element.classList.add('d-none');
	}
}

function event_status_info(status_info) {
	switch (status_info.status) {
		case "no_error":
			if (status_info.ctx === "capture_device") {
				if (overlay_msg_capture_device_in_use) {
					overlay_msg_capture_device_in_use.hide();
					overlay_msg_capture_device_in_use = null;
				}
				
				if (overlay_msg_capture_device_has_a_problem) {
					overlay_msg_capture_device_has_a_problem.hide();
					overlay_msg_capture_device_has_a_problem = null;
				}
			} else {
				if (overlay_msg_playback_device_in_use) {
					overlay_msg_playback_device_in_use.hide();
					overlay_msg_playback_device_in_use = null;
				}
				
				if (overlay_msg_playback_device_has_a_problem) {
					overlay_msg_playback_device_has_a_problem.hide();
					overlay_msg_playback_device_has_a_problem = null;
				}
			}
			break;
			
		case "device_or_resource_busy":
			if (status_info.ctx === "capture_device") {
				if (!overlay_msg_capture_device_in_use) {
					overlay_msg_capture_device_in_use = overlay_device_problem(status_info);
					overlay_msg_capture_device_in_use.show();
					close_dlg();
				}
			} else {
				if (!overlay_msg_playback_device_in_use) {
					overlay_msg_playback_device_in_use = overlay_device_problem(status_info);
					overlay_msg_playback_device_in_use.show();
					close_dlg();
				}
			}
			break;
			
		default:
			if (status_info.ctx === "capture_device") {
				if (!overlay_msg_capture_device_has_a_problem) {
					overlay_msg_capture_device_has_a_problem = overlay_device_problem(status_info);
					overlay_msg_capture_device_has_a_problem.show();
					close_dlg();
				}
			} else {
				if (!overlay_msg_playback_device_has_a_problem) {
					overlay_msg_playback_device_has_a_problem = overlay_device_problem(status_info);
					overlay_msg_playback_device_has_a_problem.show();
					close_dlg();
				}
			}
			break;
	}
}

function event_sound_card(sound_card_info) {
	if (!sound_card_info.connected) {
		if (!overlay_msg_sound_card_not_connected) {
			overlay_msg_sound_card_not_connected = overlay_sound_card_not_connected();
			overlay_msg_sound_card_not_connected.show();
			close_dlg();
		}
		return;
	} else {
		if (overlay_msg_sound_card_not_connected) overlay_msg_sound_card_not_connected.hide();
		overlay_msg_sound_card_not_connected = null;
	}
	
	if (sound_card_info.hw_mode !== "full_featured_mode") {
		if (!overlay_msg_not_full_featured_mode) {
			overlay_msg_not_full_featured_mode = overlay_not_in_full_featured_mode();
			overlay_msg_not_full_featured_mode.show();
		}
		return;
	} else {
		if (overlay_msg_not_full_featured_mode) overlay_msg_not_full_featured_mode.hide();
		overlay_msg_not_full_featured_mode = null; 
	}
}

function event_client_config_changed(changed_client_id) {
	if (client_id === changed_client_id) load_config();
}

function carousel_to(id) {
	const targetSlide = document.getElementById(id);
	const allSlides = main_content_carousel.querySelectorAll('.carousel-item');
	let targetIndex = Array.from(allSlides).indexOf(targetSlide);
	if (targetIndex === -1) return;
	
	const carousel = new bootstrap.Carousel(main_content_carousel);
	carousel.to(targetIndex);
}

function client_notification(notification) {
	if (!notification || typeof notification !== "object") return;
	if (window.location.hostname !== 'localhost' && window.location.hostname !== '127.0.0.1') return;	
	
	if ("display" in notification) {
		switch(notification.display) {
			case "next":
				set_display('on');
				new bootstrap.Carousel(main_content_carousel).next();
				break;
			
			case "prev":
				set_display('on');
				new bootstrap.Carousel(main_content_carousel).prev();
				break;
				
			case "controls":
				set_display('on');
				carousel_to("content-control-pane");
				break;
			
			case "vumeter":
				set_display('on');
				carousel_to("content-vumeter-pane");
				break;
			
			case "clock":
				set_display('on');
				carousel_to("content-clock-pane");
				break;
				
			case "off":
			case "on":
			case "toggle":
				set_display(notification.display);
				break;
		}
	}
}

// Other ===

async function subscribe_events() {
	if (!ws_do_reconnect) return;
	
	if (!overlay_msg_connecting) {
		overlay_msg_connecting = overlay_connecting();
		overlay_msg_connecting.show();
	}
	
	await wait_until_reachable();
	console.log("Connecting to WebSocket '" + ws_uri_event + "'.");
	ws_event = new WebSocket(to_ws_url(ws_uri_event));

	ws_event.onopen = async function(event) {
		console.log("Connected to WebSocket '" + ws_uri_event + "'.");
		if (!configuration_loaded) {
			await load_config();
		}
		
		if (overlay_msg_connecting) overlay_msg_connecting.hide();
		overlay_msg_connecting = null;
	};

	ws_event.onmessage = async function(event) {
		let event_as_json;
		try {
			event_as_json = JSON.parse(event.data);
		} catch (error) {
			console.error("Cannot parse json: \n" + event.data + "\n\nCause:", error);
			try { ws_event.close(); } catch (e) {}
		}

		for (let [key, handler] of Object.entries(event_handlers)) {			
			if (key in event_as_json) {
				data = event_as_json[key];
				handler(data);
				break;
			}
		}		
	}

	ws_event.onclose = async function(event) {
		ws_event = null;
		control_infos.clear();
		console.log("Connection closed to WebSocket '" + ws_uri_event + "'.");
		subscribe_events();
	};

	ws_event.onerror = async function(event) {
		try { ws_event.close(); } catch (e) {}
		console.log("Error occurred. Connection closed to WebSocket '" + ws_uri_event + "'.");
	};
	
	fetch(rest_uri_application_config, {
		method: 'GET',
		headers: { 'Accept': '*/*', 'Content-Type': 'application/json' }
	}).then(response => {
		if (response.ok) {
			response.json().then(config => {
				display_delay_ms = config.display_delay_ms;
			});
		}
	});
}

async function subscribe_vumeter() {
	if (!ws_do_reconnect) return;
	
	vumeter_enabled = true;
	
	await wait_until_reachable();
	console.log("Connecting to WebSocket '" + ws_uri_vumeter + "'.");
	ws_vumeter = new WebSocket(to_ws_url(ws_uri_vumeter));

	ws_vumeter.onopen = async function(event) {
		console.log("Connected to WebSocket '" + ws_uri_vumeter + "'.");
	};

	ws_vumeter.onmessage = async function(event) {
		let levels;
		try {
			levels = JSON.parse(event.data);
			vumeter.set_levels(levels);
		} catch (error) {
			console.error("Cannot parse json: \n" + event.data + "\n\nCause:", error);
			try { ws_vumeter.close(); } catch (e) {}
		}		
	}

	ws_vumeter.onclose = async function(event) {
		ws_vumeter = null;
		console.log("Connection closed to WebSocket '" + ws_uri_vumeter + "'.");
		if (vumeter_enabled) subscribe_vumeter();
	};

	ws_vumeter.onerror = async function(event) {
		try { ws_vumeter.close(); } catch (e) {}
		console.log("Error occurred. Connection closed to WebSocket '" + ws_uri_vumeter + "'.");
	};
}

function assign_config(node, configuration) {
	settings_menu_map.forEach((control_info, key) => {
		if (control_info.hasOwnProperty("setter")) {
			const id = control_info.id.replace(/\//g, "_");
			if (configuration.hasOwnProperty(id)) {			
				control_info.value = configuration[id];
				control_info.setter();
			}
		}
	});
}

async function load_config() {
	const response = await fetch(rest_uri_client_configuration + "?id=" + client_id);
    if (response.ok) {
		const configuration = await response.json();
		assign_config(settings_menu, configuration);
	}
	configuration_loaded = true;
	client.classList.remove('d-none');
}

function set_ui_theme() {
	const control_info = settings_menu_map.get(settings_ui_theme_show_dark);
	if (control_info.value) {  
		document.documentElement.setAttribute('data-bs-theme','dark');
	} else {
		document.documentElement.setAttribute('data-bs-theme','light');
	}
	
	const styles = getComputedStyle(document.documentElement);
	const bg_color = styles.getPropertyValue('--bs-body-bg').trim();
	change_theme_color(bg_color);
}

function set_clock_show_24h() {
	const control_info = settings_menu_map.get(settings_ui_clock_show_24h);
	if (clock) clock.show_24h = (control_info.value === 1) ? true : false;
}

function set_clock_show_seconds() {
	const control_info = settings_menu_map.get(settings_ui_clock_show_seconds);
	if (clock) clock.show_seconds = (control_info.value === 1) ? true : false;
}

function set_clock_show_date() {
	const control_info = settings_menu_map.get(settings_ui_clock_show_date);
	if (clock) clock.show_date = (control_info.value === 1) ? true : false;
}

function process_settings_menu(node, parent) {
	if (node.type === 'include') return;
	node.parent = parent;
	node.change_listeners = [];
	settings_menu_map.set(node.id, node);	
 
	if (node.hasOwnProperty("children")) {
		for (const child of node.children) process_settings_menu(child, node);
	}
}

// Connect main controls to events =====================================================

if (source) {
	source.addEventListener('click', open_switch_source_dlg);
}

if (main_volume_slider) {
	add_slider_listeners(main_volume_slider, main_volume_id);
	main_volume_slider.addEventListener('slide', (e) => {
		if (!e.detail.is_sliding) {
			hide_volume_overlay();
			
			const control_info = control_infos.get(main_volume_id);			
			send_control_value(control_info);
		}
	});
}

if (main_volume_mute) {
	main_volume_mute.addEventListener('click', function() {		
		let control_info = control_infos.get(main_volume_mute_id);
		if (!control_info) return;
		control_info.value = control_info.value == 1 ? 0 : 1;

		for (const listener of control_info.change_listeners) listener(control_info.value, false);
		
		send_control_value(control_info);
	});
}

// Dialog functions ================================================================================

function dlg_create(dom_header, dom_body, additional_body_classes) {
	const result = document.createElement("div");
	result.className = "dlg";
	
	result.appendChild(dom_header);

	const dlg_body = document.createElement("div");
	dlg_body.className = "dlg-body";
	result.appendChild(dlg_body);
	
	const fade_top = document.createElement("div");
	fade_top.className = "dlg-body-fade-top";
	dlg_body.appendChild(fade_top);
	
	const dlg_body_content = document.createElement("div");
	dlg_body_content.className = "dlg-body-content" + (additional_body_classes ? " " + additional_body_classes : "");
	if (Array.isArray(dom_body)) {
		for (const child of dom_body) dlg_body_content.appendChild(child);
	} else dlg_body_content.appendChild(dom_body);
	dlg_body.appendChild(dlg_body_content);

	const fade_bottom = document.createElement("div");
	fade_bottom.className = "dlg-body-fade-bottom";
	dlg_body.appendChild(fade_bottom);
		
	return result;
}

function dlg_create_header_btn(caption, icon_classes, align_left, onclick) {
	const result = document.createElement("button");
	if (align_left) result.className = "dlg-header-btn left";
	else result.className = "dlg-header-btn right";
	result.setAttribute("onclick", onclick);
	
	if (icon_classes && align_left) {
		const icon = document.createElement("i"); 
		icon.className = "icon " + icon_classes;
		result.appendChild(icon);
	}
	
	if (caption && caption.trim() !== '') {
		const dom_caption = document.createElement("span");
		dom_caption.className = "caption";
		dom_caption.innerHTML = caption;
		result.appendChild(dom_caption);
	}
	
	if (icon_classes && !align_left) {
		const icon = document.createElement("i"); 
		icon.className = "icon " + icon_classes;
		result.appendChild(icon);
	}
	
	const fuzzy_touch = document.createElement("div");
	fuzzy_touch.className = "dlg-header-fuzzy-touch";
	fuzzy_touch.setAttribute("onclick", onclick);
	result.appendChild(fuzzy_touch);
	
	return result;
}

function dlg_create_header_dummy_btn() {
	const result = document.createElement("div");
	result.className = "dlg-header-btn";
	return result;	
}

function dlg_create_header(caption, dom_btn_left, dom_btn_right) {
	const dom_result = document.createElement("div");
	dom_result.className = "dlg-header";
	
	if (dom_btn_left) dom_result.appendChild(dom_btn_left);
	else dom_result.appendChild(dlg_create_header_dummy_btn());
	
	const dom_caption = document.createElement("div");
	dom_caption.className = "dlg-header-caption";
	dom_caption.innerHTML = caption;
	dom_result.appendChild(dom_caption);
	
	if (dom_btn_right) dom_result.appendChild(dom_btn_right);
	else dom_result.appendChild(dlg_create_header_dummy_btn());
		
	return dom_result;
}

function dlg_create_group(caption, items) {
	const dom_result = document.createElement("div");
	dom_result.className = "dlg-group";
	
	if (caption && typeof caption === 'string' && caption.length > 0) {
		const dom_header = document.createElement("div");
		dom_header.className = "dlg-group-header";
		dom_header.textContent = caption;
		dom_result.appendChild(dom_header);
	}
	
	const dom_body = document.createElement("div");
	dom_body.className = "dlg-group-body";
	dom_result.appendChild(dom_body);
	
	if (items) {
		if (Array.isArray(items)) {
			let add_separator = false;
			for (const item of items) {
				if (add_separator) {
					const dom_sep = document.createElement("div");
					dom_sep.className = "dlg-group-sep";
					dom_body.appendChild(dom_sep);
				}
				dom_body.appendChild(item);
				add_separator = true;
			}
		} else dom_body.appendChild(items);
	}
	
	return dom_result;
}

function dlg_create_group_item(content, active) {
	const dom_result = document.createElement("div");
	if (active) dom_result.className = "dlg-group-item active";
	else dom_result.className = "dlg-group-item";
		
	if (content) {
		if (Array.isArray(content)) {
			for (const item of content) dom_result.appendChild(item);
		} else {
			if (typeof content === 'string') dom_result.textContent = content;
			else dom_result.appendChild(content);
		}
	}
	
	return dom_result;
}

// Overlay callbacks ===============================================================================

function snapshot_settings_listeners() {
	for (const control_info of control_infos.values()) {
		if (control_info.category === "settings") control_info.listeners_snapshot = control_info.change_listeners.length;
	}
}

function remove_settings_listeners() { 
	for (const control_info of control_infos.values()) {
		if (control_info.category === "settings") {
			while (control_info.change_listeners.length > control_info.listeners_snapshot) control_info.change_listeners.pop();
		}
	}
}

function slide_to_settings_menu(id, forward, select_item) {
	if (select_item) {
		const item = document.getElementById(select_item);
		if (item) item.classList.add('active');		
	}
	
	remove_settings_listeners();
	
	dlg_carousel.slide_to(render_settings_dlg(id), forward);
}

function render_menu_group(control_info) {
	let menu_items = [];
	let result = [];
	if ('children' in control_info) {
		for (const child of control_info.children) {
			if (child.type === 'menu') {
				menu_items.push(create_control(child, 1));			
			} else {
				if (menu_items.length > 0 && child.type !== 'menu') {
					result.push(dlg_create_group(null, menu_items));
					menu_items = [];
				}
				result.push(create_control(child, 1));
			}
		}
	}
	if (menu_items.length > 0) {
		result.push(dlg_create_group(null, menu_items));			
	}
	return result;
}

function render_settings_dlg(id) {
	let menu = settings_menu;
	if (settings_menu_map.has(id)) menu = settings_menu_map.get(id); 
	
	// Header
	let header_btn_left = null;
	if (menu.parent) header_btn_left = dlg_create_header_btn(menu.parent.label, "hi hi-chevron-left", true, "slide_to_settings_menu('" + menu.parent.id + "', false)");
	const header_btn_right = dlg_create_header_btn(null, "bi bi-x-lg", false, "close_dlg()");

	const header = dlg_create_header(menu.label, header_btn_left, header_btn_right);
	
	// Body
	snapshot_settings_listeners();
	let body = render_menu_group(menu);
	
	return dlg_create(header, body);
}

function open_settings(id) {
	if (!dlg_overlay) return;
	dlg_carousel.set_slide(render_settings_dlg(id));
	
	dlg_overlay.style.opacity = '1';
	dlg_overlay.style.visibility = 'visible';
}

function update_source_list(value) {
	let group = document.getElementById(to_item_id(source_id));
	if (group) {
		const old_selection = group.querySelector('.dlg-list-item.active');
		if (old_selection) old_selection.classList.remove('active');
		
		const new_selection = document.getElementById(to_value_id(source_id, value));
		if (new_selection) new_selection.classList.add('active');	
	}
}

function switch_source_to(value) {
	update_source_list(value);
	
	let control_info = control_infos.get(source_id);
	control_info.value = value;
	let new_source = get_translated_value(control_info, value);
	if (source) source.innerHTML = new_source;
	
	send_control_value(control_info);
	
	setTimeout(close_dlg, 300);		
}

function create_source_list_item(index, value_item, active) {
	let result = document.createElement('div');
	result.className = "dlg-list-item dlg-group-item" + (active ? " active" : "");
	result.setAttribute('id', to_value_id(source_id, value_item.value));
	result.setAttribute('onclick',"switch_source_to(" + value_item.value + ")");
	
	const no = document.createElement('span');
	no.textContent = index;
	no.className = "dlg-list-item-no";
	result.appendChild(no);
	
	const label = document.createElement('span');
	label.textContent = value_item.label;
	result.appendChild(label);
		
	return result;
}

function render_switch_source() {	
	// Header
	let header_btn_left = null;
	//TODO header_btn_left = dlg_create_header_btn("Rename", "hi bi-pencil-square", true, "slide_to_settings_menu('', false)");
	const header_btn_right = dlg_create_header_btn(null, "bi bi-x-lg", false, "close_dlg()");

	const header = dlg_create_header("Switch Source", header_btn_left, header_btn_right);
	
	// Body
	let control_info = control_infos.get(source_id);
	let items = [];
	let index = control_info.min;
	let no = 1;
	for (let value_item of control_info.value_labels) {
		items.push(create_source_list_item(no++, value_item, control_info.value === index++));
	}
	let body = dlg_create_group(null, items);
	body.setAttribute("id", to_item_id(control_info.id));
	
	return dlg_create(header, body, "source");
}

function open_switch_source_dlg() {
	if (!dlg_overlay) return;
	
	let control_info;
	let changed = false;
	for (let i = 0; i < 8; i++) {
		const id = signal_generator_output_to_id + "/" + i;
		control_info = control_infos.get(id);
		if (control_info.value === 1) changed = true;		
		control_info.value = 0;		
		send_control_value(control_info);
	}

	if (changed) {
		update_control_view();
		return;
	}
	
	dlg_carousel.set_slide(render_switch_source());
	
	dlg_overlay.style.opacity = '1';
	dlg_overlay.style.visibility = 'visible';
}

let closing_dlg = false;

function close_dlg() {	
	if (!dlg_overlay) return;
	
	var computed_style = window.getComputedStyle(dlg_overlay); 
	if (computed_style.opacity === '0') return;
	
	closing_dlg = true;
	dlg_overlay.style.opacity = '0';
	dlg_overlay.style.visibility = 'hidden';
	
	remove_settings_listeners();
}

if (dlg_overlay) {
	dlg_overlay.addEventListener('transitionend', (e) => {
		if (closing_dlg && e.propertyName === 'opacity') {
			dlg_carousel.remove_slides();
			closing_dlg = false;
		}
	});
}

function switch_display_to(target, elem) {
	elem.classList.add("active");
	
	fetch(rest_uri_client_notification, {
		method: 'POST',
		headers: { 'Accept': '*/*', 'Content-Type': 'application/json' },
		body: '{"display": "' + target + '"}'
	});
		
	setTimeout(close_dlg, 300);		
}

function create_display_list_item(icon_cls, caption, target) {
	let result = document.createElement('div');
	result.className = "dlg-list-item dlg-group-item";
	result.setAttribute('onclick',"switch_display_to('" + target + "', this)");
	
	const icon = document.createElement('span');
	icon.className = "dlg-list-item-icon";
	icon.className = icon_cls;
	result.appendChild(icon);
		
	const label = document.createElement('span');	
	label.textContent = caption;
	result.appendChild(label);
	return result;
}

function render_display() {	
	// Header
	const header_btn_right = dlg_create_header_btn(null, "bi bi-x-lg", false, "close_dlg()");
	const header = dlg_create_header("Display on DSC", null, header_btn_right);
	
	// Body
	let items = [];
	items.push(create_display_list_item("bi bi-sliders2", "Controls & Status", "controls"));
	items.push(create_display_list_item("bi bi-speedometer2", "VU Meter", "vumeter"));
	items.push(create_display_list_item("bi bi-clock", "Clock", "clock"));
	items.push(create_display_list_item("hi hi-display-off", "Display off", "off"));	
	let body = dlg_create_group(null, items);
	
	return dlg_create(header, body, "source");
}

function open_display_dlg() {
	if (!dlg_overlay) return;
	dlg_carousel.set_slide(render_display());
	
	dlg_overlay.style.opacity = '1';
	dlg_overlay.style.visibility = 'visible';
}

// Overlay Message =====================================================

function overlay_message(msg, hint, zindex, bouncing_dots, delay) {
	const STATES = {
		UNINITIALIZED : 0,
		SHOW_TRRIGGERED: 1,
		FADE_IN: 2,
	    SHOWING: 3,		
		FADE_OUT: 4,
		HIDDEN: 5
	};
	const fade_duration = 300; // Must match CSS
	let state = STATES.UNINITIALIZED;
	let main;
	let hint_id = 'overlay-hint-' + Math.random();
	
	if (bouncing_dots) main = msg + '&nbsp;<span class="dot-bounce dot-1">.</span><span class="dot-bounce dot-2">.</span><span class="dot-bounce dot-3">.</span>';
	else main = msg;
	
	let overlay = document.createElement('div');
	overlay.setAttribute('class', 'overlay-system-message');
	overlay.setAttribute('style', 'z-index: ' + (zindex + 2000000) + ';'); // TODO verbessern 
	overlay.innerHTML = `<div>
	        <div class='msg-main'>${main}</div>
	        <div id='${hint_id}' class='msg-hint'>${hint}</div>
	    </div>`;
	
	return {
		show: function() {
			if (state !== STATES.UNINITIALIZED) return;			
			state = STATES.SHOW_TRRIGGERED;						
			document.body.appendChild(overlay);
			
			setTimeout(function() {
				if (state !== STATES.SHOW_TRRIGGERED) return;
				console.log("Showing overlay: " + msg);
				state = STATES.FADE_IN;
				overlay.classList.add('visible');
				setTimeout(function() {
					STATES.SHOWING;					
				}, fade_duration);
			}, delay);
			
			setTimeout(function() {
	            let hint = document.getElementById(hint_id);
				if (hint) hint.classList.add('visible');
	        }, 4000);
		},
		
		hide: function() {
			switch (state) {
				case STATES.SHOW_TRRIGGERED:
				case STATES.UNINITIALIZED:
					state = STATES.HIDDEN;
					document.body.removeChild(overlay);
					break;
				
				case STATES.FADE_IN:
				case STATES.SHOWING:
					state = STATES.FADE_OUT;					
					console.log("Hiding overlay: " + msg);
					overlay.classList.remove('visible');
					
					setTimeout(function() {
						state = STATES.HIDDEN;
						document.body.removeChild(overlay);
						}, 300);
					break;
					
				default:					
					return;	
			}
		},
		
		remove: function() {
			document.body.removeChild(overlay);
		}
    };
}

function overlay_connecting() {
	return overlay_message(
		'Connecting to DSC', 
		`<ul>
	        <li>Switch on the DSC device.</li>
			<li>If the device is already restarting, just wait.</li>
	        <li>Start the DSC application on the device (if it does not start automatically).</li>
	        <li>Make sure that the network connection is working.</li>			
	    </ul>`, 1000, true, 300);
}

function overlay_sound_card_not_connected() {
	return overlay_message(
		'The sound card is not connected.',
		`<ul>
			<li>Please check the usb connection between sound card and host.</li>
			<li>Further help can be found in the troubleshooting section of the sound card manual.</li>
		</ul>`, 900, false, 0);
}

function overlay_not_in_full_featured_mode() {
	return overlay_message(
		'The sound card is not in the full-featured mode.', 
		`<ul>
	        <li>Move the switch to 'Full Mode' on the sound card. Then, turn the power off and on.</li>
	        <li>Further details can be found in the manual of the sound card.</li>
	    </ul>`, 800, false, 0);
	
}

function overlay_device_problem(status_info) {
	let device_name;
	let msg;
	let hint;
	let zindex = 700;
	switch (status_info.ctx) {
		case "capture_device":
			device_name = "capture device";
			zindex += 50;
			break;
			
		case "playback_device":
			device_name = "playback device";
			break;
			
		default:
			console.log("Unsupported device problem.");
			return;
	}
	
	switch (status_info.status) {
		case "no_error":
			return null;
			
		case "device_or_resource_busy":
			zindex += 10;
			msg = "The " + device_name + " is in use.";
			hint = `Following process has aquired the ${device_name}:
					<ul>
				        <li>Process ID: ${status_info.pid}</li>
				        <li>Command Line: ${status_info.path}</li>
				    </ul>`;
			break;
			
		default:
			msg = "There is a problem with the " + device_name + ".";
			hint = "This should never happen. Anyhow, let's turn the power off and on of the device."
			break;
	}
	
	return overlay_message(msg, hint, zindex, false, 0);
	
}

// MAIN ================================================================

// Suppress pinch-to-zoom
document.addEventListener('touchstart', function(event) {
    if (event.touches.length > 1) event.preventDefault();
}, { passive: false });

// Disable reconnect of websockets
window.addEventListener('unload', function () {
	console.log("Leaving page");
	ws_do_reconnect = false;
	return true;
});

// On opening dropdown menu: set max menu height and scroll to selected item 
document.addEventListener('shown.bs.dropdown', function (event) {
	// Set max menu height
	const menu_button = event.target;
	const menu_dropdown = event.target.nextElementSibling;
	const rect = menu_button.getBoundingClientRect();

	const spaceAbove = rect.top;
	const spaceBelow = window.innerHeight - rect.bottom;

	const maxHeight = Math.max(spaceAbove, spaceBelow) - 16;
	menu_dropdown.style.maxHeight = `${maxHeight}px`;
  	
	// Select active item
	const dropdown_menu = event.target.nextElementSibling;
	const active_item = dropdown_menu.querySelector('.active');
	if (active_item) active_item.scrollIntoView({ block: 'center', behavior: 'instant' });
});

// Disable selection
document.body.style.userSelect = 'none';
document.body.style.webkitUserSelect = 'none';
document.body.style.mozUserSelect = 'none';
document.body.style.msUserSelect = 'none';

// Alternate menu for iPhone
if (navigator.standalone) {
	client.classList.add('standalone');	
}

// Work-around for devices with incorrect implementation of the units vw and vh
function set_css_dimensions() {
	const style = document.documentElement.style;
	style.setProperty('--vpw', window.innerWidth + 'px');
	style.setProperty('--vph', window.innerHeight + 'px');
}

window.addEventListener('resize', set_css_dimensions);

document.addEventListener('DOMContentLoaded', () => { 
	set_css_dimensions();
	
	// Work-around: Swipe does not work after loading page
	bootstrap.Carousel.getOrCreateInstance(main_content_carousel); 
	 
	if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
		const brightness = control_infos.get(settings_ui_brightness_id);
		if (!brightness || brightness.visible)
			document.getElementById("display_off").classList.remove('d-none');
    } else {
        document.getElementById("display_dlg").classList.remove('d-none');
    }
});

// Before and after crousel slides		
main_content_carousel.addEventListener('slide.bs.carousel', function (event) { // Before
	// Update button indicators on swiping
	const buttons = document.querySelectorAll('#content-indicators button');
	buttons[event.from].classList.remove('active');
	buttons[event.to].classList.add('active');
	
	switch (event.to) {
		case 1:
			vumeter.start();
			subscribe_vumeter();
			break;
			
		case 2:
			clock.start();
			break;
	} 
});

main_content_carousel.addEventListener('slid.bs.carousel', function (event) { // After
	switch (event.from) {
		case 1:
			vumeter_enabled = false;
			ws_vumeter.close();
			vumeter.stop();
			break;
			
		case 2:
			clock.stop();
			break;
	}
});

// Subscribe events
subscribe_events();

// Process settings menu
process_settings_menu(settings_menu, null);

// Log CSS variables
//document.addEventListener('DOMContentLoaded', () => { log_css_variables(); });